## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 5
)

## ----eval=FALSE---------------------------------------------------------------
# install.packages("autoFlagR")

## -----------------------------------------------------------------------------
library(autoFlagR)
library(dplyr)

## -----------------------------------------------------------------------------
# Example healthcare data
data <- data.frame(
  patient_id = 1:200,
  age = rnorm(200, 50, 15),
  cost = rnorm(200, 10000, 5000),
  length_of_stay = rpois(200, 5),
  gender = sample(c("M", "F"), 200, replace = TRUE),
  diagnosis = sample(c("A", "B", "C"), 200, replace = TRUE)
)

# Introduce some anomalies
data$cost[1:5] <- data$cost[1:5] * 20  # Unusually high costs
data$age[6:8] <- c(200, 180, 190)  # Impossible ages

# Prepare data for anomaly detection
prepared <- prep_for_anomaly(data, id_cols = "patient_id")

## -----------------------------------------------------------------------------
# Score anomalies using Isolation Forest
scored_data <- score_anomaly(
  data, 
  method = "iforest", 
  contamination = 0.05
)

# View anomaly scores
head(scored_data[, c("patient_id", "anomaly_score")], 10)

## -----------------------------------------------------------------------------
# Flag top anomalies
flagged_data <- flag_top_anomalies(
  scored_data, 
  contamination = 0.05
)

# View flagged anomalies
anomalies <- flagged_data[flagged_data$is_anomaly, ]
head(anomalies[, c("patient_id", "anomaly_score", "is_anomaly")], 10)

## ----eval=FALSE---------------------------------------------------------------
# # Generate PDF report (saves to tempdir() by default)
# generate_audit_report(
#   data,
#   filename = "my_audit_report",
#   output_dir = tempdir(),
#   output_format = "pdf",
#   method = "iforest",
#   contamination = 0.05
# )

