/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.RamUsageEstimator;

public final class PagedBytes
implements Accountable {
    private static final long BASE_RAM_BYTES_USED = RamUsageEstimator.shallowSizeOfInstance(PagedBytes.class);
    private byte[][] blocks = new byte[16][];
    private int numBlocks;
    private final int blockSize;
    private final int blockBits;
    private final int blockMask;
    private boolean didSkipBytes;
    private boolean frozen;
    private int upto;
    private byte[] currentBlock;
    private final long bytesUsedPerBlock;
    private static final byte[] EMPTY_BYTES = new byte[0];

    public PagedBytes(int blockBits) {
        assert (blockBits > 0 && blockBits <= 31) : blockBits;
        this.blockSize = 1 << blockBits;
        this.blockBits = blockBits;
        this.blockMask = this.blockSize - 1;
        this.upto = this.blockSize;
        this.bytesUsedPerBlock = RamUsageEstimator.alignObjectSize(this.blockSize + RamUsageEstimator.NUM_BYTES_ARRAY_HEADER);
        this.numBlocks = 0;
    }

    private void addBlock(byte[] block) {
        if (this.blocks.length == this.numBlocks) {
            this.blocks = (byte[][])Arrays.copyOf(this.blocks, ArrayUtil.oversize(this.numBlocks, RamUsageEstimator.NUM_BYTES_OBJECT_REF));
        }
        this.blocks[this.numBlocks++] = block;
    }

    public void copy(IndexInput in, long byteCount) throws IOException {
        while (byteCount > 0L) {
            int left2 = this.blockSize - this.upto;
            if (left2 == 0) {
                if (this.currentBlock != null) {
                    this.addBlock(this.currentBlock);
                }
                this.currentBlock = new byte[this.blockSize];
                this.upto = 0;
                left2 = this.blockSize;
            }
            if ((long)left2 < byteCount) {
                in.readBytes(this.currentBlock, this.upto, left2, false);
                this.upto = this.blockSize;
                byteCount -= (long)left2;
                continue;
            }
            in.readBytes(this.currentBlock, this.upto, (int)byteCount, false);
            this.upto = (int)((long)this.upto + byteCount);
            break;
        }
    }

    public void copy(BytesRef bytes2, BytesRef out) {
        int left2 = this.blockSize - this.upto;
        if (bytes2.length > left2 || this.currentBlock == null) {
            if (this.currentBlock != null) {
                this.addBlock(this.currentBlock);
                this.didSkipBytes = true;
            }
            this.currentBlock = new byte[this.blockSize];
            this.upto = 0;
            left2 = this.blockSize;
            assert (bytes2.length <= this.blockSize);
        }
        out.bytes = this.currentBlock;
        out.offset = this.upto;
        out.length = bytes2.length;
        System.arraycopy(bytes2.bytes, bytes2.offset, this.currentBlock, this.upto, bytes2.length);
        this.upto += bytes2.length;
    }

    public Reader freeze(boolean trim2) {
        if (this.frozen) {
            throw new IllegalStateException("already frozen");
        }
        if (this.didSkipBytes) {
            throw new IllegalStateException("cannot freeze when copy(BytesRef, BytesRef) was used");
        }
        if (trim2 && this.upto < this.blockSize) {
            byte[] newBlock = new byte[this.upto];
            System.arraycopy(this.currentBlock, 0, newBlock, 0, this.upto);
            this.currentBlock = newBlock;
        }
        if (this.currentBlock == null) {
            this.currentBlock = EMPTY_BYTES;
        }
        this.addBlock(this.currentBlock);
        this.frozen = true;
        this.currentBlock = null;
        return new Reader(this);
    }

    public long getPointer() {
        if (this.currentBlock == null) {
            return 0L;
        }
        return (long)this.numBlocks * (long)this.blockSize + (long)this.upto;
    }

    @Override
    public long ramBytesUsed() {
        long size = BASE_RAM_BYTES_USED + RamUsageEstimator.shallowSizeOf((Object[])this.blocks);
        if (this.numBlocks > 0) {
            size += (long)(this.numBlocks - 1) * this.bytesUsedPerBlock;
            size += RamUsageEstimator.sizeOf(this.blocks[this.numBlocks - 1]);
        }
        if (this.currentBlock != null) {
            size += RamUsageEstimator.sizeOf(this.currentBlock);
        }
        return size;
    }

    @Override
    public Collection<Accountable> getChildResources() {
        return Collections.emptyList();
    }

    public long copyUsingLengthPrefix(BytesRef bytes2) {
        if (bytes2.length >= 32768) {
            throw new IllegalArgumentException("max length is 32767 (got " + bytes2.length + ")");
        }
        if (this.upto + bytes2.length + 2 > this.blockSize) {
            if (bytes2.length + 2 > this.blockSize) {
                throw new IllegalArgumentException("block size " + this.blockSize + " is too small to store length " + bytes2.length + " bytes");
            }
            if (this.currentBlock != null) {
                this.addBlock(this.currentBlock);
            }
            this.currentBlock = new byte[this.blockSize];
            this.upto = 0;
        }
        long pointer = this.getPointer();
        if (bytes2.length < 128) {
            this.currentBlock[this.upto++] = (byte)bytes2.length;
        } else {
            this.currentBlock[this.upto++] = (byte)(0x80 | bytes2.length >> 8);
            this.currentBlock[this.upto++] = (byte)(bytes2.length & 0xFF);
        }
        System.arraycopy(bytes2.bytes, bytes2.offset, this.currentBlock, this.upto, bytes2.length);
        this.upto += bytes2.length;
        return pointer;
    }

    public PagedBytesDataInput getDataInput() {
        if (!this.frozen) {
            throw new IllegalStateException("must call freeze() before getDataInput");
        }
        return new PagedBytesDataInput();
    }

    public PagedBytesDataOutput getDataOutput() {
        if (this.frozen) {
            throw new IllegalStateException("cannot get DataOutput after freeze()");
        }
        return new PagedBytesDataOutput();
    }

    static /* synthetic */ byte[] access$802(PagedBytes x0, byte[] x1) {
        x0.currentBlock = x1;
        return x1;
    }

    public final class PagedBytesDataOutput
    extends DataOutput {
        @Override
        public void writeByte(byte b) {
            if (PagedBytes.this.upto == PagedBytes.this.blockSize) {
                if (PagedBytes.this.currentBlock != null) {
                    PagedBytes.this.addBlock(PagedBytes.this.currentBlock);
                }
                PagedBytes.access$802(PagedBytes.this, new byte[PagedBytes.this.blockSize]);
                PagedBytes.this.upto = 0;
            }
            ((PagedBytes)PagedBytes.this).currentBlock[((PagedBytes)PagedBytes.this).upto++] = b;
        }

        @Override
        public void writeBytes(byte[] b, int offset, int length) {
            int left2;
            assert (b.length >= offset + length);
            if (length == 0) {
                return;
            }
            if (PagedBytes.this.upto == PagedBytes.this.blockSize) {
                if (PagedBytes.this.currentBlock != null) {
                    PagedBytes.this.addBlock(PagedBytes.this.currentBlock);
                }
                PagedBytes.access$802(PagedBytes.this, new byte[PagedBytes.this.blockSize]);
                PagedBytes.this.upto = 0;
            }
            int offsetEnd = offset + length;
            while (true) {
                left2 = offsetEnd - offset;
                int blockLeft = PagedBytes.this.blockSize - PagedBytes.this.upto;
                if (blockLeft >= left2) break;
                System.arraycopy(b, offset, PagedBytes.this.currentBlock, PagedBytes.this.upto, blockLeft);
                PagedBytes.this.addBlock(PagedBytes.this.currentBlock);
                PagedBytes.access$802(PagedBytes.this, new byte[PagedBytes.this.blockSize]);
                PagedBytes.this.upto = 0;
                offset += blockLeft;
            }
            System.arraycopy(b, offset, PagedBytes.this.currentBlock, PagedBytes.this.upto, left2);
            PagedBytes.this.upto += left2;
        }

        public long getPosition() {
            return PagedBytes.this.getPointer();
        }
    }

    public final class PagedBytesDataInput
    extends DataInput {
        private int currentBlockIndex;
        private int currentBlockUpto;
        private byte[] currentBlock;

        PagedBytesDataInput() {
            this.currentBlock = PagedBytes.this.blocks[0];
        }

        @Override
        public PagedBytesDataInput clone() {
            PagedBytesDataInput clone = PagedBytes.this.getDataInput();
            clone.setPosition(this.getPosition());
            return clone;
        }

        public long getPosition() {
            return (long)this.currentBlockIndex * (long)PagedBytes.this.blockSize + (long)this.currentBlockUpto;
        }

        public void setPosition(long pos) {
            this.currentBlockIndex = (int)(pos >> PagedBytes.this.blockBits);
            this.currentBlock = PagedBytes.this.blocks[this.currentBlockIndex];
            this.currentBlockUpto = (int)(pos & (long)PagedBytes.this.blockMask);
        }

        @Override
        public byte readByte() {
            if (this.currentBlockUpto == PagedBytes.this.blockSize) {
                this.nextBlock();
            }
            return this.currentBlock[this.currentBlockUpto++];
        }

        @Override
        public void readBytes(byte[] b, int offset, int len) {
            int left2;
            int blockLeft;
            assert (b.length >= offset + len);
            int offsetEnd = offset + len;
            while ((blockLeft = PagedBytes.this.blockSize - this.currentBlockUpto) < (left2 = offsetEnd - offset)) {
                System.arraycopy(this.currentBlock, this.currentBlockUpto, b, offset, blockLeft);
                this.nextBlock();
                offset += blockLeft;
            }
            System.arraycopy(this.currentBlock, this.currentBlockUpto, b, offset, left2);
            this.currentBlockUpto += left2;
        }

        private void nextBlock() {
            ++this.currentBlockIndex;
            this.currentBlockUpto = 0;
            this.currentBlock = PagedBytes.this.blocks[this.currentBlockIndex];
        }
    }

    public static final class Reader
    implements Accountable {
        private static final long BASE_RAM_BYTES_USED = RamUsageEstimator.shallowSizeOfInstance(Reader.class);
        private final byte[][] blocks;
        private final int blockBits;
        private final int blockMask;
        private final int blockSize;
        private final long bytesUsedPerBlock;

        private Reader(PagedBytes pagedBytes) {
            this.blocks = (byte[][])Arrays.copyOf(pagedBytes.blocks, pagedBytes.numBlocks);
            this.blockBits = pagedBytes.blockBits;
            this.blockMask = pagedBytes.blockMask;
            this.blockSize = pagedBytes.blockSize;
            this.bytesUsedPerBlock = pagedBytes.bytesUsedPerBlock;
        }

        public void fillSlice(BytesRef b, long start, int length) {
            assert (length >= 0) : "length=" + length;
            assert (length <= this.blockSize + 1) : "length=" + length;
            b.length = length;
            if (length == 0) {
                return;
            }
            int index2 = (int)(start >> this.blockBits);
            int offset = (int)(start & (long)this.blockMask);
            if (this.blockSize - offset >= length) {
                b.bytes = this.blocks[index2];
                b.offset = offset;
            } else {
                b.bytes = new byte[length];
                b.offset = 0;
                System.arraycopy(this.blocks[index2], offset, b.bytes, 0, this.blockSize - offset);
                System.arraycopy(this.blocks[1 + index2], 0, b.bytes, this.blockSize - offset, length - (this.blockSize - offset));
            }
        }

        public void fill(BytesRef b, long start) {
            int index2 = (int)(start >> this.blockBits);
            b.bytes = this.blocks[index2];
            byte[] block = b.bytes;
            int offset = (int)(start & (long)this.blockMask);
            if ((block[offset] & 0x80) == 0) {
                b.length = block[offset];
                b.offset = offset + 1;
            } else {
                b.length = (block[offset] & 0x7F) << 8 | block[1 + offset] & 0xFF;
                b.offset = offset + 2;
                assert (b.length > 0);
            }
        }

        @Override
        public long ramBytesUsed() {
            long size = BASE_RAM_BYTES_USED + RamUsageEstimator.shallowSizeOf((Object[])this.blocks);
            if (this.blocks.length > 0) {
                size += (long)(this.blocks.length - 1) * this.bytesUsedPerBlock;
                size += RamUsageEstimator.sizeOf(this.blocks[this.blocks.length - 1]);
            }
            return size;
        }

        @Override
        public Collection<Accountable> getChildResources() {
            return Collections.emptyList();
        }

        public String toString() {
            return "PagedBytes(blocksize=" + this.blockSize + ")";
        }
    }
}

