(ns comms.producer
  (:require [clojure.core.async :refer [go chan <! >! close!]]
            [clojure.spec.alpha :as spec]
            [clojure.core.match :refer [match]]
            [taoensso.nippy :refer [freeze]]
            [comms.util :refer [as-properties]])
  (:import [org.apache.kafka.clients.producer KafkaProducer ProducerRecord]
           [org.apache.kafka.common.serialization Serializer]))


(def ^{:private true} serializer
  (reify Serializer
    (configure [_ _ _])
    (serialize [_ _ data] (freeze data))
    (close [_])))


(defn- default-options
  ([server]
   {:bootstrap-servers server
    :key-serializer (.getName (class serializer))
    :value-serializer (.getName (class serializer))})
  ([server id] (assoc (default-options server) :transactional-id id)))


(spec/def ::record (spec/keys :req-un [::topic ::key ::value]))


(spec/def ::records (spec/coll-of ::record))


(spec/def ::input (spec/or :record ::record
                           :records ::records))


(defn- send-record [producer {:keys [topic key value]}]
  (.send producer (ProducerRecord. topic key value)))


(defn- send-records [producer records]
  (doseq [record records] (send-record producer record)))


(defn- send-input [producer conformed]
  (match conformed
         [:record record] (send-record producer record)
         [:records records] (send-records producer records)
         :else nil))


(defn- cleanup [producer errors]
  (.close producer)
  (when errors (close! errors)))


(defn create [{:keys [server options errors]}]
  (let [input-channel (chan)
        final-options (merge (default-options server) options)]
    (go
      (let [producer (KafkaProducer. (as-properties final-options))]
        (loop []
          (let [input (<! input-channel)
                conformed (spec/conform ::input input)]
            (cond
              (nil? input)
              (cleanup producer errors)

              (= :clojure.spec.alpha/invalid conformed)
              (do (when errors (>! errors (spec/explain-data input)))
                  (recur)) 

              :else
              (do (send-input producer conformed)
                  (recur)))))))
    input-channel))


(defn create-transactional [{:keys [id server options errors]
                             :or {id (gensym)}}]
  (let [input-channel (chan)
        final-options (merge (default-options server id) options)]
    (go
      (let [producer (KafkaProducer. (as-properties final-options))]
        (.initTransactions producer)
        (loop []
          (let [input (<! input-channel)
                conformed (spec/conform ::input input)]
            (cond
              (nil? input)
              (cleanup producer errors)

              (= :clojure.spec.alpha/invalid conformed)
              (do (when errors (>! errors (spec/explain-data input)))
                  (recur)) 

              :else
              (do (try
                    (.beginTransaction producer)
                    (send-input producer conformed)
                    (.commitTransaction producer)
                    (catch Exception _ (.abortTransaction producer)))
                  (recur)))))))
    input-channel))
