(ns comms.consumer
  (:require [clojure.core.async :refer [chan go >!]]
            [taoensso.nippy :refer [thaw]]
            [comms.util :refer [as-properties]])
  (:import [org.apache.kafka.clients.consumer KafkaConsumer]
           [org.apache.kafka.common.serialization Deserializer]))


(def ^{:private true} deserializer
  (reify Deserializer
    (configure [_ _ _])
    (deserialize [_ _ data] (thaw data))
    (close [_])))


(defn- default-options [server id]
  {:bootstrap-servers server
   :group-id id
   :enable-auto-commit "true"
   :auto-offset-reset "earliest"
   :key-deserializer (.getName (class deserializer))
   :value-deserializer (.getName (class deserializer))})


(defn create [{:keys [id server topics options poll-frequency]
               :or {id (gensym) poll-frequency 100}}]
  (let [output-channel (chan)
        final-options (merge (default-options server id) options)]
    (go
      (let [consumer (KafkaConsumer. (as-properties final-options))]
        (.subscribe consumer topics)
        (loop []
          (let [records (.poll consumer poll-frequency)]
            (doseq [record records]
              (>! output-channel {:value (.value record)
                                  :key (.key record)
                                  :topic (.topic record)
                                  :partition (.partition record)
                                  :offset (.offset record)})))
          (recur))))
    output-channel))

