(ns flow.core
  (:require [clojure.core.async :refer [go go-loop chan <! >! timeout
                                        alt! close! put!]]
            [clojure.core.match :refer [match]]
            [clojure.spec :as s]
            [clojure.spec.test :as test]
            [taoensso.nippy :as nippy :refer [freeze thaw]])
  (:import [kafka.utils ZkUtils]
           [kafka.admin AdminUtils]
           [org.apache.kafka.common.serialization Serializer Deserializer]
           [org.apache.kafka.common TopicPartition]
           [org.apache.kafka.clients.producer KafkaProducer ProducerRecord]
           [org.apache.kafka.clients.consumer KafkaConsumer]
           [scala.collection JavaConversions]))


(s/fdef as-properties
        :args (s/cat :hash-map map?)
        :ret #(= java.util.Properties (class %)))


(defn- as-properties [hash-map]
  (let [properties (java.util.Properties.)]
    (doseq [[key value] hash-map]
      (.setProperty properties
                    (clojure.string/replace (name key) #"-" ".")
                    value))
    properties))


(s/def ::zookeeper #(= kafka.utils.ZkUtils (class %)))
(s/def ::session-timeout int?)
(s/def ::connection-timeout int?)
(s/fdef connect-to-zookeeper
        :args (s/or :without-config (s/cat :connection-string string?)
                    :with-config (s/cat :connection-string string?
                                        :config (s/keys :opt-un [::session-timeout
                                                                 ::connection-timeout])))
        :ret ::zookeeper)


(defn connect-to-zookeeper
  ([connection-string] (connect-to-zookeeper connection-string {}))
  ([connection-string {:keys [session-timeout connection-timeout]
                       :or {session-timeout 10000 connection-timeout 10000}}]
   (let [pairs (fn [x] [(._1 x) (._2 x)])
         [client connection] (pairs (ZkUtils/createZkClientAndConnection
                                     connection-string
                                     session-timeout
                                     connection-timeout))]
     (ZkUtils. client connection false))))


(s/def ::partitions int?)
(s/def ::replication-factor int?)
(s/def ::config map?)
(s/def ::rack-aware-mode #(boolean (#{:disabled :enforced :safe} %)))
(s/fdef create-topic
        :args (s/or :without-config (s/cat :zookeeper ::zookeeper
                                           :name string?)
                    :with-config (s/cat :zookeeper ::zookeeper
                                        :name string?
                                        :config (s/keys :opt-un [::partitions
                                                                 ::replication-factor
                                                                 ::config
                                                                 ::rack-aware-mode]))))


(defn create-topic
  ([zookeeper name] (create-topic zookeeper name {}))
  ([zookeeper name {:keys [partitions replication-factor config rack-aware-mode]
                    :or {partitions 1 replication-factor 1 config nil rack-aware-mode :safe}}]
   (AdminUtils/createTopic zookeeper
                           name
                           (int partitions)
                           (int replication-factor)
                           (as-properties config)
                           (case rack-aware-mode
                             :disabled (kafka.admin.RackAwareMode$Disabled$.)
                             :enforced (kafka.admin.RackAwareMode$Enforced$.)
                             :safe (kafka.admin.RackAwareMode$Safe$.)))))


(s/fdef topics
        :args ::zookeeper
        :ret set?)


(defn topics [zookeeper]
  (-> (.getAllTopics zookeeper)
      JavaConversions/seqAsJavaList
      set))


(s/fdef delete-topic
        :args (s/cat :zookeeper ::zookeeper :name string?))


(defn delete-topic [zookeeper name]
  (AdminUtils/deleteTopic zookeeper name))


(def serializer
  (reify Serializer
    (configure [_ _ _])
    (serialize [_ _ data] (freeze data))
    (close [_])))


(def deserializer
  (reify Deserializer
    (configure [_ _ _])
    (deserialize [_ _ data] (thaw data))
    (close [_])))


(s/def ::sequential-or-string (s/or :sequential sequential?
                                    :string string?))

(s/fdef join-if-sequential
        :args ::sequential-or-string
        :ret string?)


(defn- join-if-sequential [xs]
  (if (sequential? xs)
    (clojure.string/join "," xs)
    xs))


(s/def ::producer #(= org.apache.kafka.clients.producer.KafkaProducer (class %)))
(s/fdef create-producer
        :args (s/or :servers ::sequential-or-string
                    :with-config (s/cat :servers ::sequential-or-string
                                        :config ::config))
        :ret ::producer)


(defn create-producer
  ([servers] (create-producer servers {}))
  ([servers config]
   (-> {:bootstrap-servers (join-if-sequential servers)}
       (merge config)
       as-properties
       (KafkaProducer. serializer serializer))))


(s/def ::freezable #(nippy/freezable? %))
(s/def ::topic string?)
(s/def ::partition int?)
(s/def ::timestamp #(= java.lang.Long (class %)))
(s/def ::key ::freezable)
(s/def ::value ::freezable)
(s/fdef send-to
        :args (s/or :no-key (s/cat :producer ::producer
                                   :topic ::topic
                                   :value ::value)
                    :no-partition (s/cat :producer ::producer
                                         :topic ::topic
                                         :key ::key
                                         :value ::value)
                    :no-timestamp (s/cat :producer ::producer
                                         :topic ::topic 
                                         :partition ::partition
                                         :key ::key
                                         :value ::value)
                    :full (s/cat :producer ::producer
                                 :topic ::topic 
                                 :partition ::partition
                                 :timestamp ::timestamp
                                 :key ::key
                                 :value ::value)))


(defn send-to
  ([producer topic value]
   (.send producer (ProducerRecord. topic value)))
  ([producer topic key value]
   (.send producer (ProducerRecord. topic key value)))
  ([producer topic partition key value]
   (.send producer (ProducerRecord. topic partition key value)))
  ([producer topic partition timestamp key value]
   (.send producer (ProducerRecord. topic partition timestamp key value))))


(s/def ::consumer #(= org.apache.kafka.clients.consumer.KafkaConsumer (class %)))
(s/fdef create-consumer
        :args ::sequential-or-string
        :ret ::consumer)


(defn- create-consumer [servers]
  (-> {:bootstrap-servers (join-if-sequential servers)
       :group-id (str (gensym)) 
       :auto-offset-reset "earliest"}
      as-properties
      (KafkaConsumer. deserializer deserializer)))


(s/def ::channel #(= clojure.core.async.impl.channels.ManyToManyChannel (class %)))
(s/def ::events ::channel)
(s/def ::commands ::channel)
(s/def ::command vector?)
(s/def ::subscriptions set?)
(s/fdef process-command
        :args (s/cat :consumer ::consumer
                     :events ::events
                     :commands ::commands
                     :command ::command
                     :subscriptions ::subscriptions)
        :ret ::subscriptions)


(defn- process-command [consumer events commands command subscriptions]
  (match command
         [:subscribe & topics]
         (do (.subscribe consumer topics)
             (into subscriptions topics))

         [:poll]
         (do (when-not (empty? subscriptions)
               (doseq [event (.poll consumer 100)]
                 (put! events {:value (.value event)
                               :key (.key event)
                               :topic (.topic event)
                               :partition (.partition event)
                               :offset (.offset event)})))
             subscriptions)

         [:seek-to position topic partition]
         (do (when (subscriptions topic)
               (case position
                 :beginning (.seekToBeginning consumer [(TopicPartition. topic partition)])))
             subscriptions)))


(s/fdef consumer->channel
        :args ::sequential-or-string
        :ret (s/cat :commands ::channel :events ::channel))


(defn flow-consumer [servers]
  (let [events (chan 1000)
        commands (chan 10)
        new-timer #(timeout 100)]
    (go (let [consumer (create-consumer servers)
              process-command (partial process-command consumer events commands)]
          (loop [subscriptions #{}
                 timer (new-timer)]
            (alt!
              commands ([command]
                        (if (nil? command)
                          (do (.close consumer) 
                              (close! events))
                          (recur (process-command command subscriptions) timer)))


              timer (do (>! commands [:poll])
                        (recur subscriptions (new-timer)))))))
    {:commands commands
     :events events}))

