(ns flow.core
  (:require [clojure.core.async :refer [go go-loop chan <! >! timeout
                                        alt! close! put!]]
            [clojure.core.match :refer [match]]
            [clojure.spec :as s]
            [clojure.spec.test :as test]
            [taoensso.nippy :as nippy :refer [freeze thaw]])
  (:import [kafka.utils ZkUtils]
           [kafka.admin AdminUtils]
           [org.apache.kafka.common.serialization Serializer Deserializer]
           [org.apache.kafka.common TopicPartition]
           [org.apache.kafka.clients.producer KafkaProducer ProducerRecord]
           [org.apache.kafka.clients.consumer KafkaConsumer]
           [scala.collection JavaConversions]))


(defn is-class [c]
  (fn [x]
    (= c (class x))))


(s/fdef as-properties
        :args (s/cat :hash-map map?)
        :ret (is-class java.util.Properties))


(defn- as-properties [hash-map]
  (let [properties (java.util.Properties.)]
    (doseq [[key value] hash-map]
      (.setProperty properties
                    (clojure.string/replace (name key) #"-" ".")
                    (str value)))
    properties))


(s/def ::zookeeper (is-class kafka.utils.ZkUtils))
(s/def ::session-timeout-ms int?)
(s/def ::connection-timeout-ms int?)
(s/def ::zookeeper-config (s/keys :opt-un [::session-timeout-ms ::connection-timeout-ms]))


(def default-zookeeper-config
  {:session-timeout-ms 10000
   :connection-timeout-ms 10000})


(s/fdef connect-to-zookeeper
        :args (s/or :without-config (s/cat :connection-string string?)
                    :with-config (s/cat :connection-string string?
                                        :config ::zookeeper-config))
        :ret ::zookeeper)


(defn connect-to-zookeeper
  ([connection-string] (connect-to-zookeeper connection-string {}))
  ([connection-string config]
   (let [{:keys [session-timeout-ms connection-timeout-ms]} (merge default-zookeeper-config config)
         pairs (fn [x] [(._1 x) (._2 x)])
         [client connection] (pairs (ZkUtils/createZkClientAndConnection
                                     connection-string
                                     session-timeout-ms
                                     connection-timeout-ms))]
     (ZkUtils. client connection false))))


(defn long? [x] (instance? Long x))
(s/def ::partitions int?)
(s/def ::replication-factor int?)
(s/def ::rack-aware-mode #{:disabled :enforced :safe})
(s/def ::segment-bytes int?)
(s/def ::segment-ms long?)
(s/def ::segment-index-bytes long?)
(s/def ::flush-messages long?)
(s/def ::flush-ms long?)
(s/def ::retention-bytes long?)
(s/def ::retention-ms long?)
(s/def ::max-message-bytes int?)
(s/def ::index-interval-bytes int?)
(s/def ::delete-retention-ms long?)
(s/def ::file-delete-delay-ms long?)
(s/def ::min-cleanable-dirty-ratio double?)
(s/def ::cleanup-policy string?)


(s/def ::topic-config (s/keys :opt-un [::partitions ::replication-factor ::rack-aware-mode
                                       ::segment-bytes ::segment-ms ::segment-index-bytes
                                       ::flush-messages ::flush-ms ::retention-bytes ::retention-ms
                                       ::max-message-bytes ::index-interval-bytes ::delete-retention-ms
                                       ::file-delete-delay-ms ::min-cleanable-dirty-ratio
                                       ::cleanup-policy]))


(def default-topic-config
  {:partitions 1
   :replication-factor 1
   :rack-aware-mode :safe
   :segment-bytes 1048576
   :segment-ms (Long/MAX_VALUE)
   :segment-index-bytes 1048576
   :flush-messages (Long/MAX_VALUE)
   :flush-ms (Long/MAX_VALUE)
   :retention-bytes -1
   :retention-ms -1
   :max-message-bytes (Integer/MAX_VALUE)
   :index-interval-bytes 4096
   :delete-retention-ms 86400000
   :file-delete-delay-ms 60000 
   :min-cleanable-dirty-ratio 0.5
   :cleanup-policy "delete"})



(s/fdef create-topic
        :args (s/or :without-config (s/cat :zookeeper ::zookeeper
                                           :name string?)
                    :with-config (s/cat :zookeeper ::zookeeper
                                        :name string?
                                        :config ::topic-config)))


(defn create-topic
  ([zookeeper name] (create-topic zookeeper name {}))
  ([zookeeper name config]
   (let [{:keys [partitions replication-factor rack-aware-mode]} (merge default-topic-config config)]
     (AdminUtils/createTopic zookeeper
                             name
                             (int partitions)
                             (int replication-factor)
                             (as-properties (dissoc config :partitions :replication-factor :rack-aware-mode))
                             (case rack-aware-mode
                               :disabled (kafka.admin.RackAwareMode$Disabled$.)
                               :enforced (kafka.admin.RackAwareMode$Enforced$.)
                               :safe (kafka.admin.RackAwareMode$Safe$.))))))


(s/fdef topics
        :args (s/cat :zookeeper ::zookeeper)
        :ret set?)


(defn topics [zookeeper]
  (-> (.getAllTopics zookeeper)
      JavaConversions/seqAsJavaList
      set))


(s/fdef delete-topic
        :args (s/cat :zookeeper ::zookeeper
                     :name string?))


(defn delete-topic [zookeeper name]
  (AdminUtils/deleteTopic zookeeper name))


(def ^:private serializer
  (reify Serializer
    (configure [_ _ _])
    (serialize [_ _ data] (freeze data))
    (close [_])))


(def ^:private deserializer
  (reify Deserializer
    (configure [_ _ _])
    (deserialize [_ _ data] (thaw data))
    (close [_])))


(s/def ::sequential-or-string (s/or :sequential sequential?
                                    :string string?))

(s/fdef join-if-sequential
        :args (s/cat :xs ::sequential-or-string)
        :ret string?)


(defn- join-if-sequential [xs]
  (if (sequential? xs)
    (clojure.string/join "," xs)
    xs))


(s/def ::producer (is-class org.apache.kafka.clients.producer.KafkaProducer))
(s/def ::serializer #(instance? Serializer %))
(s/def ::key-serializer ::serializer)
(s/def ::value-serializer ::serializer)
(s/def ::acks #{-1 0 1 "all"})
(s/def ::buffer-memory long?)
(s/def ::compression-type #{"none" "gzip" "snappy" "lz4"})
(s/def ::retries int?)
(s/def ::batch-size int?)
(s/def ::client-id string?)
(s/def ::connections-max-idle-ms long?)
(s/def ::linger-ms long?)
(s/def ::max-block-ms long?)
(s/def ::max-request-size int?)
(s/def ::receive-buffer-bytes int?)
(s/def ::request-timeout-ms int?)
(s/def ::security-protocol #{"PLAINTEXT" "SSL" "SASL_PLAIN-TEXT" "SASL_SSL"})
(s/def ::send-buffer-bytes int?)
(s/def ::ssl-keystore-type string?)
(s/def ::ssl-protocol #{"TLS" "TLSv1.1" "TLSv1.2" "SSL" "SSLv2" "SSLv3"})
(s/def ::ssl-truststore-type string?)
(s/def ::timeout-ms int?)
(s/def ::block-on-buffer-full boolean?)
(s/def ::max-in-flight-requests-per-connection int?)
(s/def ::metadata-fetch-timeout-ms long?)
(s/def ::metadata-max-age-ms long?)
(s/def ::metrics-num-samples int?)
(s/def ::metrics-sample-window-ms long?)
(s/def ::reconnect-backoff-ms long?)
(s/def ::retry-backoff-ms long?)
(s/def ::sasl-kerberos-kinit-cmd string?)
(s/def ::sasl-kerberos-min-time-before-relogin long?)
(s/def ::sasl-kerberos-ticket-renew-jitter double?)
(s/def ::sasl-kerberos-ticket-renew-window-factor double?)
(s/def ::ssl-keymanager-algorithm string?)
(s/def ::ssl-trustmanager-algorithm string?)
(s/def ::producer-config (s/keys :opt-un [::key-serializer ::value-serializer
                                          ::acks ::buffer-memory ::compression-type
                                          ::retries ::batch-size ::client-id
                                          ::connections-max-idle-ms ::linger-ms
                                          ::max-block-ms ::max-request-size
                                          ::receive-buffer-bytes ::request-timeout-ms
                                          ::security-protocol ::send-buffer-bytes
                                          ::ssl-keystore-type ::ssl-protocol
                                          ::ssl-truststore-type ::timeout-ms
                                          ::block-on-buffer-full
                                          ::max-in-flight-requests-per-connection
                                          ::metadata-fetch-timeout-ms ::metadata-max-age-ms
                                          ::metrics-num-samples ::metrics-sample-window-ms
                                          ::reconnect-backoff-ms ::rety-backoff-ms
                                          ::sasl-kerberos-kinit-cmd ::sasl-kerberos-min-time-before-relogin
                                          ::sasl-kerberos-ticket-renew-jitter
                                          ::sasl-kerberos-service-name ::sasl-kerberos-ticket-renew-jitter
                                          ::sasl-kerberos-ticket-renew-window-factor
                                          ::ssl-keymanager-algorithm ::ssl-trustmanager-algorithm]))


(def default-producer-config
  {:key-serializer serializer
   :value-serializer serializer
   :acks "all"
   :buffer-memory 33554432
   :compression-type "none"
   :reties 0
   :batch-size 16384
   :client-id ""
   :connections-max-idle-ms 540000
   :linger-ms 0
   :max-block-ms 60000
   :max-request-size 1048576
   :receive-buffer-bytes 32768
   :request-timeout-ms 30000
   :security-protocol "PLAINTEXT"
   :send-buffer-bytes 131072
   :ssl-keystore-type "JKS"
   :ssl-protocol "TLS"
   :ssl-truststore-type "JKS"
   :timeout-ms 30000
   :block-on-buffer-full false
   :max-in-flight-requests-per-connection 5
   :metadata-fetch-timeout-ms 60000
   :metadata-max-age-ms 300000
   :metrics-num-samples 2
   :metrics-sample-window-ms 30000
   :reconnect-backoff-ms 50
   :retry-backoff-ms 100
   :sasl-kerberos-kinit-cmd "/usr/bin/kinit"
   :sasl-kerberos-min-time-before-relogin 60000
   :sasl-kerberos-ticket-renew-jitter 0.05
   :sasl-kerberos-ticket-renew-window-factor 0.8
   :ssl-keymanager-algorithm "SunX509"
   :ssl-trustmanager-algorithm "PKIX"})


(s/fdef create-producer
        :args (s/or :without-config (s/cat :servers ::sequential-or-string)
                    :with-config (s/cat :servers ::sequential-or-string
                                        :config ::producer-config))
        :ret ::producer)


(defn- create-producer
  ([servers] (create-producer servers {}))
  ([servers config]
   (let [config (merge default-producer-config config)
         {:keys [key-serializer value-serializer]} config]
     (-> config
         (dissoc :key-serializer :value-serializer) 
         (assoc :bootstrap-servers (join-if-sequential servers))
         as-properties
         (KafkaProducer. key-serializer value-serializer)))))


(s/def ::any (fn [_] true))
(s/def ::topic string?)
(s/def ::partition int?)
(s/def ::timestamp #(= java.lang.Long (class %)))
(s/def ::key ::any)
(s/def ::value ::any)
(s/fdef send-to
        :args (s/or :no-key (s/cat :producer ::producer
                                   :topic ::topic
                                   :value ::value)
                    :no-partition (s/cat :producer ::producer
                                         :topic ::topic
                                         :key ::key
                                         :value ::value)
                    :no-timestamp (s/cat :producer ::producer
                                         :topic ::topic 
                                         :partition ::partition
                                         :key ::key
                                         :value ::value)
                    :full (s/cat :producer ::producer
                                 :topic ::topic 
                                 :partition ::partition
                                 :timestamp ::timestamp
                                 :key ::key
                                 :value ::value)))


(defn- send-to
  ([producer topic value]
   (.send producer (ProducerRecord. topic value)))
  ([producer topic key value]
   (.send producer (ProducerRecord. topic key value)))
  ([producer topic partition key value]
   (.send producer (ProducerRecord. topic partition key value)))
  ([producer topic partition timestamp key value]
   (.send producer (ProducerRecord. topic partition timestamp key value))))


(s/def ::channel (is-class clojure.core.async.impl.channels.ManyToManyChannel))
(s/fdef flow-producer
        :args (s/or :without-config (s/cat :servers ::sequential-or-string)
                    :with-config (s/cat :servers ::sequential-or-string
                                        :config ::producer-config))
        :ret ::channel)


(defn flow-producer
  ([servers] (flow-producer servers default-producer-config))
  ([servers config]
   (let [input (chan 1000)]
     (go (let [producer (create-producer servers config)
               send-to (partial send-to producer)]
           (loop []
             (when-let [message (<! input)]
               (if (nil? message)
                 (.close producer)
                 (do (apply send-to message)
                     (recur)))))))
     input)))


(s/def ::consumer (is-class org.apache.kafka.clients.consumer.KafkaConsumer))
(s/def ::deserializer #(instance? Deserializer %))
(s/def ::key-deserializer ::deserializer)
(s/def ::value-deserializer ::deserializer)
(s/def ::fetch-min-bytes int?)
(s/def ::group-id string?)
(s/def ::heartbeat-interval-ms int?)
(s/def ::max-partition-fetch-bytes int?)
(s/def ::auto-offset-reset #{"latest" "earliest" "none"})
(s/def ::enable-auto-commit boolean?)
(s/def ::exclude-internal-topics boolean?)
(s/def ::fetch-max-bytes int?)
(s/def ::max-poll-interval-ms int?)
(s/def ::max-poll-records int?)
(s/def ::partition-assignment-strategy string?)
(s/def ::sasl-mechanism string?)
(s/def ::auto-commit-interval-ms int?)
(s/def ::check-crcs boolean?)
(s/def ::fetch-max-wait-ms int?)


(def default-consumer-config
  {:key-deserializer deserializer
   :value-deserializer deserializer
   :fetch-min-bytes 1
   :group-id "default"
   :heartbeat-interval-ms 3000
   :max-partition-fetch-bytes 1048576
   :session-timeout-ms 10000
   :auto-offset-reset "earliest"
   :connections-max-idle-ms 540000
   :enable-auto-commit true
   :exclude-internal-topics true
   :fetch-max-bytes 52428800
   :max-poll-interval-ms 300000
   :max-poll-records 500
   :partition-assignment-strategy "org.apache.kafka.clients.consumer.RangeAssignor"
   :receive-buffer-bytes 65536
   :request-timeout-ms 305000
   :sasl-mechanism "GSSAPI"
   :security-protocol "PLAINTEXT"
   :send-buffer-bytes 131072
   :ssl-keystore-type "JKS"
   :ssl-protocol "TLS"
   :ssl-truststore-type "JKS"
   :auto-commit-interval-ms 5000
   :check-crcs true
   :client-id ""
   :fetch-max-wait-ms 500
   :metadata-max-age-ms 300000
   :metrics-num-samples 2
   :metrics-sample-window-ms 30000
   :reconnect-backoff-ms 50
   :retry-backoff-ms 100
   :sasl-kerberos-kinit-cmd "/usr/bin/kinit"
   :sasl-kerberos-min-time-before-relogin 60000
   :sasl-kerberos-ticket-renew-jitter 0.05
   :sasl-kerberos-ticket-renew-window-factor 0.8
   :ssl-keymanager-algorithm "SunX509"
   :ssl-trustmanager-algorithm "PKIX"})


(s/def ::consumer-config (s/keys :opt-un [::key-deserializer ::value-deserializer
                                          ::fetch-min-bytes ::group-id ::heartbeat-interval-ms
                                          ::max-partition-fetch-bytes ::session-timeout-ms
                                          ::auto-offset-reset ::connections-max-idle-ms
                                          ::enable-auto-commit ::exclude-internal-topics
                                          ::fetch-max-bytes ::max-poll-interval-ms
                                          ::max-poll-records ::partition-assignment-strategy
                                          ::receive-buffer-bytes ::request-timeout-ms
                                          ::sasl-mechanism ::security-protocol
                                          ::send-buffer-bytes ::ssl-keystore-type
                                          ::ssl-protocol ::ssl-truststore-type
                                          ::auto-commit-interval-ms ::check-crcs
                                          ::client-id ::fetch-max-wait-ms
                                          ::metadata-max-age-ms ::metrics-num-samples
                                          ::metrics-sample-window-ms ::reconnect-backoff-ms
                                          ::retry-backoff-ms ::sasl-kerberos-kinit-cmd
                                          ::sasl-kerberos-min-time-before-relogin
                                          ::sasl-kerberos-ticket-renew-jitter
                                          ::sasl-kerberos-ticket-renew-window-factor
                                          ::ssl-keymanager-algorithm ::ssl-trustmanager-algorithm]))


(s/fdef create-consumer
        :args (s/or :without-config (s/cat :servers ::sequential-or-string
                                           :topics coll?)
                    :with-config (s/cat :servers ::sequential-or-string
                                        :topics coll?
                                        :config ::producer-config))
        :ret ::consumer)


(defn- create-consumer
  ([servers topics] (create-consumer servers topics {}))
  ([servers topics config]
   (let [config (merge default-consumer-config config)
         {:keys [key-deserializer value-deserializer]} config
         consumer (-> config
                      (assoc :bootstrap-servers (join-if-sequential servers))
                      as-properties
                      (KafkaConsumer. key-deserializer value-deserializer))]
     (.subscribe consumer topics)
     consumer)))


(s/fdef poll-consumer
        :args (s/cat :consumer ::consumer)
        :ret list?)


(defn- poll-consumer [consumer]
  (map
   (fn [message] {:value (.value message)
                  :key (.key message)
                  :topic (.topic message)
                  :partition (.partition message)
                  :offset (.offset message)})
   (.poll consumer 100)))


(s/fdef flow-consumer
        :args (s/or :without-config (s/cat :servers ::sequential-or-string
                                           :topics coll?)
                    :with-config (s/cat :servers ::sequential-or-string
                                        :topics coll?
                                        :config ::consumer-config))
        :ret ::channel)


(defn flow-consumer
  ([servers topics] (flow-consumer servers topics {}))
  ([servers topics config]
   (let [output (chan 1000)]
     (go (let [consumer (create-consumer servers topics config)]
           (loop [messages ()]
             (if (empty? messages)
               (recur (poll-consumer consumer))
               (do (>! output (first messages))
                   (recur (rest messages)))))))
     output)))


