(ns adzerk.clj-kinesis-producer.kinesis
  (:import
    [java.util UUID]
    [java.util.concurrent ExecutionException]
    [com.amazonaws.services.kinesis.producer
     Attempt KinesisProducerConfiguration KinesisProducer
     UserRecordFailedException UserRecordResult]))

(defn wrap-utf8-bytes
  [^String data]
  (java.nio.ByteBuffer/wrap (.getBytes data "UTF-8")))

(defn random-key
  []
  (.toString (UUID/randomUUID)))

(defn make-attempt
  [^Attempt attempt]
  {:success?      (.isSuccessful attempt)
   :delay         (.getDelay attempt)
   :duration      (.getDuration attempt)
   :error-message (.getErrorMessage attempt)
   :error-code    (.getErrorCode attempt)})

(defn make-result
  [^UserRecordResult result]
  {:success?        (.isSuccessful result)
   :sequence-number (.getSequenceNumber result)
   :shard-id        (.getShardId result)
   :attempts        (map make-attempt (.getAttempts result))})

(defprotocol IProducer
  (put-record [this stream-name partition-key record]))

(defrecord Producer [producer]
  IProducer
  (put-record [this stream-name partition-key record]
    (let [record (wrap-utf8-bytes record)
          future (.addUserRecord producer stream-name partition-key record)]
      (try (make-result (.get future))
           (catch ExecutionException ex
             (let [cause (.getCause ex)]
               (if-not (instance? UserRecordFailedException cause)
                 (throw cause)
                 (make-result (.getResult ^UserRecordFailedException cause)))))))))

(defn producer
  ([] (producer {}))
  ([options]
   (let [setters {:region                    (memfn setRegion v)
                  :credentials-provider      (memfn setCredentialsProvider v)
                  :max-connections           (memfn setMaxConnections v)
                  :request-timeout           (memfn setRequestTimeout v)
                  :record-max-buffered-time  (memfn setRecordMaxBufferedTime v)}
         setter  (fn [config k v] (when-let [s (setters k)] (s config v)) config)]
     (Producer.
       (KinesisProducer.
         (reduce-kv setter (KinesisProducerConfiguration.) options))))))
