(ns adzerk.clj-kinesis-producer.core
  (:require
    [cheshire.core :as json]
    [adzerk.clj-kinesis-producer.queue :as q]
    [adzerk.clj-kinesis-producer.worker :as w]
    [adzerk.clj-kinesis-producer.kinesis :as k]))

(defn put-record!
  [worker data]
  (w/put! worker data))

(defn start!
  [worker]
  (w/start! worker))

(defn stop!
  [worker]
  (w/stop! worker))

(defn stats
  [worker]
  (w/stats worker))

(defn kinesis-producer
  "Returns a new Kinesis producer that enqueues outbound messages locally before
  attempting to send them to Kinesis."
  [stream-name & [{:keys [dir-path on-success on-error threads]
                   :or {dir-path "/tmp/durable-queues"
                        threads 50}}]]
  (let [q (q/make-queue dir-path {:fsync-put? false
                                  :fsync-interval 100})
        k (k/producer
            {:region "us-east-1"
             :request-timeout 1000
             :record-max-buffered-time 10000})
        p #(let [rec (json/generate-string %)]
             (k/put-record k stream-name (k/random-key) rec))
        s (or on-success #(when % (prn %)))
        e (or on-error #(when % (.printStackTrace %)))]
    (w/worker q :records p s e 1000 100 threads)))

(defn simple-kinesis-producer
  "Returns a new Kinesis producer that sends records directly to Kinesis without
  enqueueing them locally first."
  [stream-name]
  (w/simple-worker stream-name))
