(ns adzerk.clj-kinesis-producer.worker
  (:require
   [cheshire.core :as json]
   [adzerk.clj-kinesis-producer.kinesis :as k]
   [adzerk.clj-kinesis-producer.queue :as queue]))

(defmacro with-callbacks
  [success error & body]
  `(try (~success (do ~@body))
        (catch Throwable t#
          (try (~error t#) (catch Throwable _#)))))

(defn num-inflight!
  [atom]
  (count (swap! atom (partial remove realized?))))

(defprotocol IWorker
  (start! [this])
  (stop!  [this])
  (stats  [this])
  (put!   [this data]))

(defrecord Worker
  [queue topic process-job on-success on-error take-timeout put-timeout max-in-flight in-flight running?]
  IWorker
  (start! [this]
    (let [sym (reset! running? (gensym))]
      (future
        (while (= @running? sym)
          (with-callbacks identity on-error
            (if (< max-in-flight (num-inflight! in-flight))
              (Thread/sleep 100)
              (swap! in-flight conj
                     (future (with-callbacks on-success on-error
                               (queue/take! queue topic take-timeout process-job))))))))))
  (stop! [this]
    (reset! running? false))
  (stats [this]
    (let [qstat (get (queue/stats queue) (name topic))]
      (merge {:in-flight (num-inflight! in-flight)} qstat)))
  (put! [this data]
    (with-callbacks identity on-error
      (queue/put! queue topic data put-timeout))))

(defrecord SimpleWorker
  [producer stream-name]
  IWorker
  (start! [this])
  (put! [this data]
    (let [rec (json/generate-string data)]
      (k/put-record producer stream-name (k/random-key) rec))))

(defn worker
  "Returns a worker that enqueues pending Kinesis messages locally/durably."
  [queue topic process-job on-success on-error take-timeout put-timeout max-in-flight]
  (Worker. queue topic process-job on-success on-error take-timeout put-timeout max-in-flight (atom ()) (atom false)))

(defn simple-worker
  "Returns a worker that sends records to Kinesis without enqueuing locally/durably."
  [stream-name]
  (let [k (k/producer
           {:region "us-east-1"
            :request-timeout 1000
            :record-max-buffered-time 10000})]
    (SimpleWorker. k stream-name)))
