(ns aft.http.core
  (:require
    [ajax.core :as ajax]
    [clojure.walk :refer [postwalk]]
    [goog.json :as goog-json]
    [camel-snake-kebab.core :as kebab]
    [aft.http.dsl :as dsl]
    ))

;; in order to consume this lib these two atoms must be set:
;; enable automatic authentication retry using jwt and localstorage
(defonce refresh-uri (atom "uninitialized"))
(defonce login-uri (atom "uninitialized"))

(declare make-request)

(defn init! [{:keys [refresh
                     login
                     on-resolution]}]
  (reset! refresh-uri refresh)
  (reset! login-uri login)
  (dsl/init! on-resolution make-request))

(defn- transform-keys
  "Recursively transform all keys in map m with transform function t."
  [t m]
  (let [f (fn [[k v]] [(t k v) v])]
    (postwalk (fn [x] (if (map? x) (into {} (map f x)) x)) m)))

(defn- json-read
  "Parse a closure XhrIo response's json into clojurescript."
  [xhrio]
  (let [json (goog-json/parse (.getResponseText xhrio) kebab/->kebab-case-keyword)]
    ;; @TODO this is super slow on large datasets: consider transit?
    (->> json
        (js->clj)
        (transform-keys kebab/->kebab-case-keyword))))

(defn- json-write [data]
  (->> data
      (transform-keys kebab/->snake_case_keyword)
      (clj->js)
      (.serialize (goog.json.Serializer.))))

(defn- json-response-format
  "Returns a JSON response format with keywordized keys
  and kebab-case rewriting of those keywords."
  []
  (ajax/map->ResponseFormat
    {:read json-read
     :content-type ["application/json"]}))

(defn- json-request-format
  "Return a JSON request format with snake_case stringified
  keys."
  []
  {:write json-write
   :content-type "application/json"})

(defn- json-patch-request-format
  "Returns a JSON response format with keywordized keys
  and kebab-case rewriting of those keywords."
  []
  {:write json-write
   :content-type ["application/json-patch+json"]})

(defn- capture-refresh
  "When requesting weblogin or webrefresh, put the response's refresh token
  into localStorage for future use."
  [response]
  (if (> (.getStatus response) 299)
    response
    (let [url (.getLastUri response)
          capture? (or (re-find (re-pattern @refresh-uri) url)
                       (re-find (re-pattern @login-uri) url))]
      (when capture? (let [refresh (-> (.getResponseJson response)
                                       (js->clj :keywordize-keys true) (get :refresh))]
                       (.setItem js/localStorage "refresh" (.serialize (goog.json.Serializer.) refresh))))
      response)))

(def jwt-interceptor
  (ajax/to-interceptor {:name "jwt"
                        :response capture-refresh}))

(def default-interceptors
  [jwt-interceptor])

(defn- retry-handler
  "Retry the request repeatedly with an exponential backoff."
  [config response]
  (let [wait (atom 2000)
        retries (atom 5)]
    (letfn [(handler [[ok response]]
              (if (> (:status response) 499)
                (do (swap! wait (partial * 2))
                    (retry @wait))
                ((:handler config) response)))
            (retry [after]
              (if (> @retries 0)
                (do
                  (js/setTimeout #(request) after)
                  (swap! retries dec))
                ((:post-error-handler config) response)))
            (request []
              (ajax/ajax-request (merge config {:handler handler})))]
      (retry @wait))))

(defn- refresh [token handler]
  (ajax/ajax-request
    {:uri @refresh-uri
     :method :post
     :format (ajax/json-request-format)
     :response-format (json-response-format)
     :interceptors default-interceptors
     :headers {"authorization" (str "Bearer " token)}
     :handler handler}))

(defn- unauthorized? [response]
  (= 401 (:status response)))

(defn- with-retry [config]
  "Invoke an ajax request with the given config. If the response is 401,
  attempt to refresh the jwt token and (if successful), re-attempt the
  original request."
  (letfn [(handler [[ok response]]
            (if (> (:status response) 499)
              ((:error-handler config) config response)
              (if (unauthorized? response)
                (retry-once ok response)
                (respond (:handler config) ok response))))

          (retry-once [ok response]
            (let [token (goog-json/parse (.getItem js/localStorage "refresh"))]
              (if token
                (refresh token handle-refresh)
                (respond (:handler config) ok response))))

          (handle-refresh [[ok response]]
            (if (unauthorized? response)
              (respond (:handler config) ok response)
              (ajax/ajax-request (merge config {:handler depleted-retries}))))

          (depleted-retries [[ok response]]
            (if (> (:status response) 499)
              ((:error-handler config) config response)
              (respond (:handler config) ok response)))

          (respond [handler ok response]
            (if ok
              (handler {:response response :status 200 :failure nil})
              (handler response)))]

    (ajax/ajax-request (merge config {:handler handler}))))

(defn make-request
  "Invoke the http method with given url and args. Default arguments (such as
  json response format, auth interceptors) are provided, but can be overridden
  when needed by explicitly providing {:arg-key value} overrides. GETs will be
  automatically retried on 5xx with an exponential backoff, invoking an optional
  :error-handler fn if all retries complete unsuccessfully. Retry behavior can
  be disabled explicitly with {:retry false}."
  [{:keys [uri method] :as args}]
  (let [defaults {:uri uri
                  :method method
                  :format (if (= method :patch)
                            (json-patch-request-format)
                            (json-request-format))
                  :response-format (json-response-format)
                  :interceptors default-interceptors}
        retry (and (:retry args true)
                   (= method :get))
        post-errror-handler (or (:error-handler args) :noop)
        handlers {:handler (:handler args)
                  :error-handler (if retry retry-handler #(post-errror-handler %2))
                  :post-error-handler post-errror-handler}]
  (with-retry
    (merge defaults
           (when (= :patch (:method args)) {:format (json-patch-request-format)})
           args
           handlers))))

;; legacy support until every handler is converted into :http fx
(defn POST [url args]
  (make-request (merge args {:uri url :method :post})))

(defn GET [url args]
  (make-request (merge args {:uri url :method :get})))

(defn PATCH [url args]
  (make-request (merge args
                       {:uri url
                        :method :patch
                        :format (json-patch-request-format)})))

(defn DELETE [url args]
  (make-request (merge args {:uri url :method :delete})))

(defn PUT [url args]
  (make-request (merge args {:uri url :method :put})))
