(ns aft.http.dsl
  (:require-macros
    [aft.logging.core :refer [error]])
  )

;; the callback to invoke upon http resolution
(defonce do-http (atom #(error "aft.http.dsl was not initialized.")))
(defonce on-resolution (atom #()))

(defonce http-bounce-cache (atom {}))
(defonce http-cancellable-cache (atom {}))

(defn cfg->request
  [{:as cfg
    :keys [success failure error]
    :or {success [:http-no-success]
         failure [:http-no-failure]
         error [:http-no-error]}}]
  (letfn [(handle-success [{:keys [status response] :as r}]
            (if (> status 399)
              (@on-resolution (conj failure response r))
              (@on-resolution (conj success response r))))
          (handle-error [args]
            (@on-resolution (conj error args)))]
    (-> cfg
      (assoc :handler handle-success)
      (assoc :error-handler handle-error))))

(defn cfg->normalized-handlers
  "Provide an interface for http: success, error, failure handlers that will
  be invoked appropriately on response. Success if status < 300, error if
  299 < status < 500, failure if status > 499.

  If an :abort map with :cache (atom) and :key (uuid) is provided, a pre-handler
  will manage outstanding requests, removing them from the cache when resolved."
  [{:as cfg
    :keys [success failure error abort]
    :or {success [:http-no-success]
         failure [:http-no-failure]
         error [:http-no-error]}}]
  (letfn [(pre-handler [{:keys [status] :as args}]
            (swap! (:cache abort) dissoc (:key abort))
            (if (< status 500)
              (handle-success args)
              (handle-error args)))
          (handle-success [{:keys [status response] :as r}]
            (if (> status 399)
              (@on-resolution (conj failure response r))
              (@on-resolution (conj success response r))))
          (handle-error [args]
            (@on-resolution (conj error args)))]
    (-> cfg
      (assoc :handler pre-handler)
      (assoc :error-handler pre-handler))))

(defn- maybe-abort-request
  "Possibly abort an outstanding xhrio that exists in the given cache if the
  request has not yet resolved."
  [{:as cfg :keys [abort]}]
  (when-let [existing (get (deref (:cache abort)) (:key abort))]
    (.abort existing))
  cfg)

(defn- make-cancellable-request
  "Make an http request with the given config and store the returned xhrio
  object in the provided cache so that we can abort the xhrio in the future."
  [{:as cfg :keys [abort]}]
  (let [xhrio (@do-http cfg)]
    (swap! (:cache abort) assoc (:key abort) xhrio)))

(defn- do-request [cfg]
  (cond
    (:bounce cfg) (-> cfg
                      (assoc :abort {:cache http-bounce-cache
                                     :key (:bounce cfg)})
                      maybe-abort-request
                      cfg->normalized-handlers
                      make-cancellable-request)
    (:cancellable cfg) (-> cfg
                           (assoc :abort {:cache http-cancellable-cache
                                          :key (:cancellable cfg)})
                           maybe-abort-request
                           cfg->normalized-handlers
                           make-cancellable-request)

    (:cancel cfg) (maybe-abort-request (assoc cfg :abort {:cache http-cancellable-cache
                                                          :key (:cancel cfg)}))
    :default (-> cfg
                 cfg->request
                 (#(@do-http %)))))

;; The :http DSL api is as follows:
;;
;; {:method  > one-of :post :get :put :patch :delete
;;  :uri     > the uri (string) of the request
;;
;;  Three options for data "callbacks". On http resolution one of these three
;;  options will be invoked with the given data, plus the response data (if any)
;;  of the  http request.
;;
;;  :success > the data to return on success (x < 400)
;;  :failure > the data to return on failure (500 > x > 399)
;;  :error   > the data to return on error   (x > 499)
;;
;;  :bounce  > a uuid determining if multiple successive requests should cancel
;;             pending requests, keyed on the provided id. this is a built-in
;;             xhr abort to reduce strain on the backend. for example:
;;
;;             {:method :get
;;              :url "/api/foo"
;;              :bounce "foo-debounce"
;;              :success [:foo/resolved]
;;              :failure [:foo/errored]
;;              :error [:foo/failed]}
;;
;;  :cancellable > provide a uuid in order to allow cancellation of a request
;;                 before it returns from the server. to cancel any outstanding
;;                 request, simply generate a full :http fx with the same
;;                 :cancellable uuid (which is basically equivalent to :bounce),
;;                 or generate an :http fx only consisting of {:cancel uuid},
;;                 which will abort the outstanding xhr (if possible).
;;
;;  :cancel > see :cancellable.
;;  }
;;
;;  :success and :failure events will receive two arguments; the body's response
;;  value (if any), and the full xhr response (status, statusText, etc.).
;;
;;  :error will receive only one argument; whatever was returned from the server,
;;  if anything.
;;
;;  optionally provide a seq of maps instead of a single map to fire multiple
;;  http requests in succession.
(defn dsl [cfg]
  (cond
    (sequential? cfg) (doseq [c cfg] (dsl c))
    (map? cfg) (do-request cfg)
    (nil? cfg) :noop
    :else (error "re-frame-http-fx: expected configuration to be a list, vector, or map, but got: " cfg)))

(defn init!
  "Setup callbacks for resolution, and http invocation, return the http dsl translator."
  [cb http-method]
  (reset! on-resolution cb)
  (reset! do-http http-method)
  dsl)

