(ns aft.reframe.interceptors
  (:require
    [re-frame.core :as reframe]
    [re-frame.interceptor :as intc]
    [schema.core :as s]
    )
  (:require-macros
    [aft.logging.core :refer [log warn error group group-end]]
    )
  )

(declare truncate)
(declare truncate-event)

(defonce app-schema (atom nil))

(defn set-schema! [s]
  (reset! app-schema s))

(defn valid-schema?
  "Validate the given db, logging any problems to the console. To handle the case
  of reg-event-fx *not* returning any {:db ...} along-side their :effects, we can
  ignore nil."
  [db]
  (when db
    (let [res (s/check @app-schema db)]
      (when (some? res)
        (warn "schema problem: " res)))))

(defn path [& args]
  (apply re-frame.std-interceptors/path args))

(defn enrich [f & args]
  "A middleware factory which runs the given function f in the after handler
  position. Enrich expects f to modify the db in a useful way (unlike after
  middleware). This is best used to handle repetitive tasks, or something that
  must happen frequently for a large number of message handlers."
  (intc/->interceptor
    :name :enrich
    :after (fn enrich-after
             [context]
             (let [event (intc/get-coeffect context :event)
                   db (intc/get-effect context :db)]
               (->> (apply f db event [args])
                    (intc/assoc-effect context :db))))))

(def debug
  "Interceptor just like re-frame/debug, but only log the first N characters of
  any message. This prevents the slowdown caused by logging to console.log for
  huge datasets."
  (intc/->interceptor
    :name :debug
    :before (fn debug-before
              [context]
              (let [event (intc/get-coeffect context :event)
                    trunc (truncate-event event)]
                (log "Handling re-frame event:  " trunc)
                (intc/assoc-coeffect context :truncated-event trunc)))
    :after (fn debug-after
             [context]
             (let [event (intc/get-coeffect context :event)
                   trunc (intc/get-coeffect context :truncated-event)
                   orig-db (intc/get-coeffect context :db)
                   new-db (intc/get-effect context :db ::not-found)]
               (if (= new-db ::not-found)
                 (log "No db changes caused by:  " trunc)
                 (let [[before after] (clojure.data/diff orig-db new-db)
                       changed? (or (some? before) (some? after))]
                   (if changed?
                     (do
                       (group "db clojure.data/diff for: " trunc)
                       (log "before: " before)
                       (log "after : " after)
                       (group-end))
                     (log "No db changes after:      " trunc))))
               context))))

(def debug-lite
  "Interceptor just like debug, but doesn't log db changes, only message
  sequences. Useful for reducing log spam when not needed."
  (let [seqnum (atom 0)]
    (intc/->interceptor
      :name :debug
      :before (fn debug-before
                [context]
                (reset! seqnum (inc @seqnum))
                (let [event (intc/get-coeffect context :event)
                      trunc (truncate-event event)]
                  (log @seqnum trunc)
                  (intc/assoc-coeffect context :truncated-event trunc))))))

;; truncate large events so the dev console doesn't flood
(defn- truncate-coll
  "Recursively truncate a collection to a certain maximum length.
  Adds an ellipsis where truncation occurs."
  [coll]
  (let [max-length 5
        max-length-exceeded (> (count coll) max-length)
        coll' (if max-length-exceeded (take max-length coll) coll)]
    (cond (map? coll)
          (let [s (map (fn [[k v]] [(truncate k) (truncate v)]) coll')]
            (into {} (if max-length-exceeded
                       (concat s '([... ...]))
                       s)))
          (seq? coll)
          (let [s (map truncate coll')]
            (apply list (if max-length-exceeded
                          (concat s '(...))
                          s)))
          :else
          (let [s (map truncate coll')]
            (into (empty coll) (if max-length-exceeded
                                 (concat s '(...))
                                 s))))))

(defn- truncate-str
  "Truncate the string s if the length exceeds 20 characters and add an
  ellipsis."
  [s]
  (let [max-length 20]
    (if (> (count s) max-length)
      (str (subs s 0 max-length) "...")
      s)))

(defn- truncate
  "Truncate an object recursively."
  [x]
  (cond (coll? x) (truncate-coll x)
        (string? x) (truncate-str x)
        :else x))

(defn- truncate-event
  "Given an event vector [:event-name arg arg1... argN], trim the size of
  the args so as to not choke the logging process with enormous datasets."
  [event-vec]
  (let [event (first event-vec)
        args (rest event-vec)
        truncated (map truncate args)]
    (if (empty? args) event-vec
      (conj [event] truncated))))

(def default-interceptors [
                           (when ^:boolean goog.DEBUG debug-lite)
                           (when ^:boolean goog.DEBUG (reframe/after valid-schema?))
                           reframe/trim-v
                           ])

