(ns alda.worker
  (:require [alda.now           :as    now]
            [alda.parser        :refer (parse-input)]
            [alda.parser-util   :refer (parse-to-events-with-context)]
            [alda.lisp.score    :as    score]
            [alda.sound         :as    sound]
            [alda.sound.midi    :as    midi]
            [alda.util          :as    util]
            [alda.version       :refer (-version-)]
            [cheshire.core      :as    json]
            [clojure.core.cache :as    cache]
            [clojure.pprint     :refer (pprint)]
            [taoensso.timbre    :as    log]
            [ezzmq.core         :as    zmq]))

(def ^:dynamic *no-system-exit* false)

(defn exit!
  [exit-code]
  (when-not *no-system-exit* (System/exit exit-code)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn start-alda-environment!
  []
  (sound/start-synthesis-engine!)
  (midi/open-midi-synth!)
  (log/debug "Requiring alda.lisp...")
  (require '[alda.lisp :refer :all]))

(defn refresh-alda-environment!
  []
  (midi/open-midi-synth!))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- success-response
  [body]
  {:success true
   :body    (if (map? body)
              (json/generate-string body)
              body)})

(defn- error-response
  [e]
  {:success false
   :body    (if (string? e)
              e
              (.getMessage e))})

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def ^:const JOB_CACHE_SIZE 20)

(defrecord Job [status score error])

(def job-cache (atom (cache/fifo-cache-factory {} :threshold JOB_CACHE_SIZE)))

(defn update-job! [id job]
  "Upserts a Job record into the cache.
   If a job with that ID is already in the cache, updates it.
   Otherwise, adds it to the cache."
  (swap! job-cache (fn [c]
                     (if (cache/has? c id)
                       (-> (cache/hit c id)
                           (assoc id job))
                       (cache/miss c id job)))))

(defn update-job-status! [id status]
  "Updates the `status` of a job, preserving the existing values of `score` and
   `error`.

   If the job doesn't exist yet, adds a new job with the provided `id` and
   `status`."
  (swap! job-cache (fn [c]
                     (if (cache/has? c id)
                       (-> (cache/hit c id)
                           (update id assoc :status status))
                       (cache/miss c id (Job. status nil nil))))))

(defn pending?
  [{:keys [status] :as job}]
  (not (#{:success :error :playing} status)))

(defn available?
  []
  (not-any? #(#{:parsing :playing} (:status %)) (vals @job-cache)))

(defn run-job!
  [code {:keys [history from to jobId]}]
  (try
    (log/debugf "Starting job %s..." jobId)
    (update-job! jobId (Job. :parsing nil nil))
    (let [_ (log/debug "Parsing body...")
          [code-context code] (parse-to-events-with-context code)

          ;; If code was whitespace, normalize to ()
          code (or code ())

          ;; Parse and remove events
          _ (log/debug "Parsing history...")
          [history-context history] (parse-to-events-with-context
                                      history)

          ;; If history was whitespace, normalize to ()
          history (or history ())]
      (if-let [error (or (when (= :parse-failure code-context)
                           code)
                         (when (= :parse-failure history-context)
                           history))]
        (do
          (log/error error error)
          (update-job! jobId (Job. :error nil error)))
        (do
          (log/debug "Playing score...")
          (let [play-opts {:from     from
                           :to       to
                           :async?   true
                           :one-off? true}
                {:keys [score wait]}
                (if (empty? history)
                  (now/play-score! (score/score code) play-opts)
                  (now/with-score* (atom
                                     (-> (score/score)
                                         (score/continue history)))
                    (now/play-with-opts! play-opts code)))]
            (update-job! jobId (Job. :playing score nil))
            (wait)
            (refresh-alda-environment!))
          (log/debug "Done playing score.")
          (update-job-status! jobId :success))))
    (catch Throwable e
      (log/error e e)
      (update-job! jobId (Job. :error nil e)))))

(defn handle-code-play
  [code {:keys [jobId] :as options}]
  (-> (cond
        (empty? jobId)
        (error-response "Request missing a `jobId` option.")

        (get @job-cache jobId)
        (success-response "Already playing that score.")

        :else
        (do
          (future (run-job! code options))
          (success-response "Request received.")))
      (assoc :jobId jobId)))

(defn handle-code-parse
  [code & {:keys [mode] :or {mode :lisp}}]
  (try
    (require '[alda.lisp :refer :all])
    (success-response (case mode
                        :lisp (let [result (parse-input code mode)]
                                (with-out-str (pprint result)))
                        :map  (parse-input code mode)))
    (catch Throwable e
      (log/error e e)
      (error-response e))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmulti process :command)

(defmethod process :default
  [{:keys [command]}]
  (error-response (format "Unrecognized command: %s" command)))

(defmethod process nil
  [_]
  (error-response "Missing command"))

(defmethod process "parse"
  [{:keys [body options]}]
  (let [{:keys [as]} options]
    (case as
      "lisp" (handle-code-parse body :mode :lisp)
      "map"  (handle-code-parse body :mode :map)
      nil    (error-response "Missing option: as")
      (error-response (format "Invalid format: %s" as)))))

(defmethod process "ping"
  [_]
  (success-response "OK"))

(defmethod process "play"
  [{:keys [body options]}]
  (handle-code-play body options))

(defmethod process "play-status"
  [{:keys [options]}]
  (let [job-id (get options :jobId)
        {:keys [status score error] :as job} (get @job-cache job-id)]
    (-> (cond
          (empty? job-id)
          (error-response "Request missing a `jobId` option.")

          (nil? status)
          (error-response "Job not found.")

          (= :error status)
          (error-response error)

          :else
          (-> (success-response (name status))
              (assoc :score score)
              (assoc :pending (pending? job))))
        (assoc :jobId job-id))))

(defmethod process "version"
  [_]
  (success-response -version-))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def ^:const MIN-LIFESPAN       (* 1000 60 15)) ; 15 minutes
(def ^:const MAX-LIFESPAN       (* 1000 60 20)) ; 20 minutes
(def ^:const HEARTBEAT-INTERVAL 1000)  ; 1 second
(def ^:const SUSPENDED-INTERVAL 10000) ; 10 seconds
(def ^:const MAX-LIVES          10)

(def lives (atom MAX-LIVES))

(defn start-worker!
  [port & [verbose?]]
  (util/set-log-level! :debug)
  (when-not (or (System/getenv "ALDA_DEBUG") verbose?)
    (let [log-path (util/alda-home-path "logs" "error.log")]
      (log/infof "Logging errors to %s" log-path)
      (util/set-log-level! :error)
      (util/rolling-log! log-path)))
  (start-alda-environment!)
  (log/info "Worker reporting for duty!")
  (log/infof "Connecting to socket on port %d..." port)
  (zmq/with-new-context
    (let [running?       (atom true)
          now            (System/currentTimeMillis)
          lifespan       (+ now (rand-nth (range MIN-LIFESPAN MAX-LIFESPAN)))
          last-heartbeat (atom now)
          socket         (zmq/socket :dealer {:connect (str "tcp://*:" port)})]

      (zmq/after-shutdown
        (log/info "Shutting down."))

      (log/info "Sending READY signal.")
      (zmq/send-msg socket "READY")

      (zmq/polling {:stringify false}
        [socket :pollin [msg]
         (do
           (cond
           ; the server sends 1-frame messages as signals
           (= 1 (count msg))
           (let [signal (-> msg first (String.))]
             (case signal
               "KILL"      (do
                             (log/info "Received KILL signal from server.")
                             (reset! running? false))
               "HEARTBEAT" (do
                             (log/debug "Got HEARTBEAT from server.")
                             (reset! lives MAX-LIVES))
               (log/errorf "Invalid signal: %s" signal)))

           ; the server also forwards 3-frame messages from the client
           ; Frames:
           ;   1) the return address of the client
           ;   2) a JSON string representing the client's request
           ;   3) the command as a string (for use by the server)
           (= 3 (count msg))
           (let [[return-address body command] msg
                 body    (String. body)
                 command (String. command)]
             (try
               (when (and (not (available?))
                          (not= "play-status" command))
                 (log/debugf "Rejecting message (command: %s). I'm busy." command)
                 (throw (Exception. "The requested worker is not available.")))
               (log/debugf "Processing message... (command: %s)" command)
               (let [req (json/parse-string body true)
                     res (json/generate-string (process req))]
                 (log/debug "Sending response...")
                 (zmq/send-msg socket [return-address "" res])
                 (log/debug "Response sent."))
               (catch Throwable e
                 (log/error e e)
                 (log/info "Sending error response...")
                 (let [err (json/generate-string (error-response e))]
                   (zmq/send-msg socket [return-address "" err]))
                 (log/info "Error response sent."))))

           :else
           (log/errorf "Invalid message: %s" (mapv #(String. %) msg))))]

        (while (and (zmq/polling?) @running?)
          (let [now      (System/currentTimeMillis)
                got-msgs (zmq/poll HEARTBEAT-INTERVAL)]
            (cond
              ;; Each worker has a randomly assigned lifespan in the range of
              ;; MIN-LIFESPAN and MAX-LIFESPAN. Once this period of time has
              ;; elapsed, the worker finishes whatever work it might be doing
              ;; and then shuts down so that the server can replace it with a
              ;; fresh worker.
              ;;
              ;; This ensures that the workers available are always recently
              ;; spawned processes, which helps us avoid known audio bugs.
              (and (> now lifespan) (available?))
              (do
                (log/info "Worker lifespan elapsed. Shutting down...")
                (reset! last-heartbeat now)
                (reset! running? false))

              ;; Detect when the system has been suspended and stop working so
              ;; the server can replace it with a fresh worker.
              ;;
              ;; This fixes a bug where MIDI audio is delayed.
              (> now (+ @last-heartbeat SUSPENDED-INTERVAL))
              (do
                (log/info "Process suspension detected. Shutting down...")
                (reset! last-heartbeat now)
                (reset! running? false))

              ;; If a heartbeat wasn't received from the server within the
              ;; acceptable threshold, MAX-LIVES times in a row, conclude that
              ;; the server has stopped sending heartbeats and shut down.
              (not (contains? got-msgs 0))
              (do
                (swap! lives dec)
                (when (and (<= @lives 0) (available?))
                  (log/error "Unable to reach the server.")
                  (reset! running? false))))

            ;; Send AVAILABLE/BUSY status back to the server.
            (when (and @running?
                       (> now (+ @last-heartbeat HEARTBEAT-INTERVAL)))
              (zmq/send-msg socket (if (available?) "AVAILABLE" "BUSY"))
              (reset! last-heartbeat now)))))))
  (exit! 0))
