(ns leiningen.depgraph
  "Generate a namespace dependency graph as an svg file"
  (:use [clojure.java.shell :only [sh]]))

(def parsed-namespaces (atom #{}))

(def common-ns (atom nil))

(defn ffile [file]
  (if (string? file)
    (java.io.File. file)
    file))

(defn read-ns [file]
  (with-open [f (-> file ffile java.io.FileReader.
                java.io.PushbackReader.)]
    (binding [*in* f]
      (let [x (read)]
        (if (= (first x) 'ns)
          x
          nil)))))

(defn parse-reference-entry [ref]
  (if-not (sequential? ref)
    [ref]
    (let [ext (loop [[c & r] (rest ref), ns-ext ()]
                (cond (or (keyword? c) (nil? c)) ns-ext
                      :else (recur r (conj ns-ext (if (sequential? c)
                                                    (first c) c)))))]
      (if (empty? ext)
        [(first ref)]
        (map #(symbol (str (first ref) \. %)) ext)))))

(defn get-required [ns-form]
  (->> ns-form
       (filter #(and (coll? %) (= :require (first %))))
       (mapcat rest)
       (mapcat parse-reference-entry)
       set))

(defn get-used [ns-form]
  (->> ns-form
      (filter #(and (coll? %) (= :use (first %))))
      (mapcat rest)
      (mapcat parse-reference-entry)
      set))

(defn get-import [ns-form]
  (->> ns-form
       (filter #(and (coll? %) (= :import (first %))))
       (mapcat rest)
       (mapcat parse-reference-entry)
       set))

(defn common-namespace []
  (let [[one & other] (map #(clojure.string/split (name %) #"\.")
                           @parsed-namespaces)]
   (loop [i 0, result ""]
     (if (every? #(= (one i) (% i)) other)
       (recur (inc i) (str result (one i) "."))
       result))))

(defn parse [ns-form]
  (swap! parsed-namespaces conj (second ns-form))
  (reset! common-ns (common-namespace))
  {:name (second ns-form)
   :clojure-depends (concat (get-required ns-form)
                            (get-used ns-form))
   :java-depends (get-import ns-form)})

(defn all-clojure-files [root]
  (let [root (ffile root)]
    (if (.isDirectory root)
      (mapcat all-clojure-files
              (.listFiles root
                          (proxy [java.io.FilenameFilter] []
                            (accept [dir name]
                              (or (.endsWith name ".clj")
                                  (.isDirectory (java.io.File. dir name)))))))
      [root])))

(defn parse-directory [dir]
  (reduce #(assoc % (:name %2) (dissoc %2 :name)) {}
          (map parse (remove nil? (map read-ns (all-clojure-files dir))))))

(defn restructure [files]
  (reduce
   (fn [map- dep]
     (-> map- (update-in [:java] #(into % (:java-depends (second dep))))
         (update-in [:clojure] #(conj (into % (:clojure-depends (second dep)))
                                      (first dep)))
         (update-in [:edges]
                    #(into %
                           (map (partial vector (first dep))
                                (concat (:java-depends (second dep))
                                        (:clojure-depends (second dep))))))))
   {:java #{} :clojure #{} :edges #{}}
   files))

(def formats {:clustered "digraph simple_hierarchy {\n graph [rankdir = \"LR\"];\n %s subgraph cluster_clojure {\nlabel=\"clojure\";\ncolor=blue;\n%s} subgraph cluster_java {\nlabel=\"java\";\ncolor=red;\n%s}}"
              :sparse "digraph simple_hierarchy {\n graph [rankdir = \"LR\"];\n %s\n%s\n%s}"})

(defn dot [x fmt]
  (format (formats fmt) (:edges x) (:clojure x) (:java x)))

(defn safe-name [string]
  (str (.replaceAll (str string) "(\\.|-|\\$)" "_")))

(defn safe-name-and-label [structure tag shape multicolor only-own]
  (update-in structure [tag]
             #(clojure.string/join
               (for [x %]
                 (let [color (if (and multicolor
                                  (contains? @parsed-namespaces x))
                               "blue" "black")]
                   (if (or (not only-own) (.startsWith (str x) @common-ns))
                    (str (safe-name x) "[label=\"" x "\",shape=\"" shape "\",color=\"" color "\"];\n")))))))

(defn edges [structure only-own]
  (update-in structure [:edges]
             #(clojure.string/join
               (for [[from to] %]
                 (if (or (not only-own) (and (.startsWith (str from) @common-ns)
                                             (.startsWith (str to) @common-ns)))

                   (format "%s->%s;\n" (safe-name from) (safe-name to)))))))

(defn depgraph
  "Generate a namespace dependency graph as svg file"
  [project & args]
  (let [args (reduce #(conj % (keyword %2)) #{} args)
        source-path (:source-path project "src")
        dotfile (str (:name project) ".dot")
        svgfile (str (:name project) ".svg")]
    (println "HM?" args)
    (-> source-path
        parse-directory
        ((partial remove (comp nil? first)))
        restructure
        (safe-name-and-label :java "hexagon" (:colored args) (:only-own args))
        (safe-name-and-label :clojure "ellipse" (:colored args) (:only-own args))
        (edges (:only-own args))
        ((fn [x] (if (:clustered args) (dot x :clustered) (dot x :sparse))))
        ((partial spit dotfile)))
    (sh "dot" "-Tsvg" (str "-o" svgfile ) dotfile) ))

;; (depgraph {:name "neko", :source-path "/home/unlogic/clojure/android/neko/src"} "only-own")
