(ns leiningen.depgraph
  "Generate a namespace dependency graph as an svg file"
  (:require [dorothy.core :as dot]
            [clojure.string :as s]
            [clojure.java.io :as jio])
  (:use [clojure.java.shell :only [sh]]))

(def parsed-namespaces (atom #{}))

(def common-ns (atom ""))

(defn ffile [file]
  (if (string? file)
    (jio/file file)
    file))

(defn read-ns [file]
  (with-open [f (-> file jio/reader java.io.PushbackReader.)]
    (binding [*in* f]
      (let [x (read)]
        (when (= (first x) 'ns)
          x)))))

(defn ignored? [ns-or-class ignore-list]
  (some (fn [^String x]
          (let [lng (dec (count x))]
            (if (= (.charAt x lng) \*)
              (.startsWith (str ns-or-class) (subs x 0 lng))
              (= x (str ns-or-class))))) ignore-list))

(defn parse-reference-entry [ref]
  (if-not (sequential? ref)
    [ref]
    (let [ext (loop [[c & r] (rest ref), ns-ext ()]
                (if (or (keyword? c) (nil? c))
                  ns-ext
                  (recur r (conj ns-ext (if (sequential? c)
                                          (first c) c)))))]
      (if (empty? ext)
        [(first ref)]
        (map #(symbol (str (first ref) \. %)) ext)))))

(defn get-required [ns-form]
  (->> ns-form
       (filter #(and (coll? %) (= :require (first %))))
       (mapcat rest)
       (mapcat parse-reference-entry)
       set))

(defn get-used [ns-form]
  (->> ns-form
       (filter #(and (coll? %) (= :use (first %))))
       (mapcat rest)
       (mapcat parse-reference-entry)
       set))

(defn get-import [ns-form]
  (->> ns-form
       (filter #(and (coll? %) (= :import (first %))))
       (mapcat rest)
       (mapcat parse-reference-entry)
       set))

(defn common-namespace []
  (let [[one & other] (map #(s/split (name %) #"\.")
                           @parsed-namespaces)]
    (loop [i 0, result ""]
      (if (every? #(= (one i) (% i)) other)
        (recur (inc i) (str result (one i) "."))
        result))))

(defn parse [ns-form {:keys [ignore no-java]}]
  (swap! parsed-namespaces conj (second ns-form))
  (when-not (ignored? (second ns-form) ignore)
    {:name (second ns-form)
     :type :clojure
     :clojure-depends (concat (get-required ns-form)
                              (get-used ns-form))
     :java-depends (if no-java [] (get-import ns-form))}))

(defn java-get-classname [fname s]
  (symbol
   (str (->> (.split s "\n")
             (keep (fn [pline]
                     (if (.endsWith pline ";")
                       (re-matches #".*package ([\w\.]*);.*" pline)
                       (re-matches #".*package ([\w\.]*);.*" (apply str (butlast pline))))))
             first second)
        "." (second (re-matches #"(.+)\.java" fname)))))

(defn java-get-imports [s]
  (loop [[^String c & r] (.split s "\n")
         imports []
         ifound false]
    (if c
      (if (.startsWith (.trim c) "import")
        (recur r
               (conj imports
                     (symbol
                      (let [import-line (if (.endsWith c ";")
                                          c
                                          (apply str (butlast c)))
                            import (second (re-matches #"\s*import (.+);\s*" import-line))]
                        (if (.endsWith import ".*")
                          (second (re-matches #"(.+)\.\*" import))
                          import))))
               true)
        (if ifound
          imports
          (recur r imports false)))
      imports)))

(defn parse-java-file [f {:keys [ignore no-java]}]
  (let [contents (slurp f)
        cn (java-get-classname (.getName (ffile f)) contents)]
    (swap! parsed-namespaces conj cn)
    (when-not (or no-java (ignored? cn ignore))
      {:name cn
       :type :java
       :clojure-depends []
       :java-depends (java-get-imports contents)})))

(defn all-clojure-files [root]
  (let [root (ffile root)]
    (if (.isDirectory root)
      (mapcat all-clojure-files
              (.listFiles root
                          (proxy [java.io.FilenameFilter] []
                            (accept [dir name]
                              (or (.endsWith name ".clj")
                                  (.isDirectory (jio/file dir name)))))))
      [root])))

(defn all-java-files [root]
  (let [root (ffile root)]
    (if (.isDirectory root)
      (mapcat all-java-files
              (.listFiles root
                          (proxy [java.io.FilenameFilter] []
                            (accept [dir name]
                              (or (.endsWith name ".java")
                                  (.isDirectory (jio/file dir name)))))))
      [root])))

(defn parse-directory [dir args]
  (reduce #(assoc % (:name %2) (dissoc %2 :name)) {}
          (concat (map #(parse % args) (remove nil? (map read-ns (all-clojure-files dir))))
                  (map #(parse-java-file % args) (all-java-files dir)))))

(defn class-package [classname]
  (second (re-matches #"(.*)\..+" (name classname))))

(defn short-name [classname]
  (second (re-matches #".*\.(.+)" (name classname))))

(defn restructure [files]
  (as-> (reduce
         (fn [map- dep]
           (-> map-
               (update-in [:java] #(into % (if (= (:type (second dep)) :java)
                                             (conj (:java-depends (second dep)) (first dep))
                                             (:java-depends (second dep)))))
               (update-in [:clojure] #(into % (if (= (:type (second dep)) :clojure)
                                                (conj (:clojure-depends (second dep)) (first dep))
                                                (:clojure-depends (second dep)))))
               (update-in [:edges]
                          #(into %
                                 (map (partial vector (first dep))
                                      (concat (:java-depends (second dep))
                                              (:clojure-depends (second dep))))))))
         {:java #{} :clojure #{} :edges #{}}
         files)
        struct
        (assoc-in struct [:java-clustered]
                  (reduce (fn [m cl]
                            (if (m (class-package cl))
                              (update-in m [(class-package cl)] conj cl)
                              (assoc m (class-package cl) [cl])))
                          {} (:java struct)))))

(defn generate-graph [x {:keys [clustered title] :as args}]
  (dot/digraph :simple_hierarchy
               (concat [{:rankdir :LR
                         :label (format "%s %s" title (dissoc args :title))}]
                       (:edges x)
                       (if clustered
                         [(dot/subgraph :cluster_clojure
                                        (cons {:label "clojure"}
                                              (:clojure x)))]
                         (:clojure x))
                       (if clustered
                         [(dot/subgraph :cluster_java
                                        (cons {:label "java"}
                                              (:java x)))]
                         (:java x)))))

(defn safe-name [string]
  (str (.replaceAll (str string) "(\\.|-|\\$)" "_")))

(defn shorten-name [string]
  (let [parts (s/split string #"\.")]
    (s/join "." (conj (->> (butlast parts)
                           (mapv #(s/join "-" (map first (s/split % #"\-")))))
                      (last parts)))))

#_(shorten-name "grammarly.common.utils")
#_(shorten-name "grammarly.core-metrics.checkers.grammarly")

(defn process-clojure-nodes [structure shape {:keys [colored only-own ignore shorten]}]
  (update-in structure [:clojure]
             #(for [x %
                    :when (and (not (ignored? x ignore))
                               (or (not only-own) (.startsWith (str x) @common-ns)))
                    :let [color (if (and colored (contains? @parsed-namespaces x))
                                  "blue" "black")]]
                [(keyword x) {:label (if (and shorten (contains? @parsed-namespaces x))
                                       (shorten-name (str x))
                                       (str x))
                              :shape shape, :color color}])))

(defn process-java-nodes [structure shape {:keys [colored only-own ignore shorten group-java]}]
  (assoc structure :java
         ((if group-java identity (partial apply concat))
          (for [[package classes] (:java-clustered structure)
                :let [classes-nodes
                      (for [x classes
                            :when (and (not (ignored? x ignore))
                                       (or (not only-own) (.startsWith (str x) @common-ns)))
                            :let [color (if (and colored (.startsWith (str x) @common-ns))
                                          "blue" "black")]]
                        [(keyword x) {:label (cond group-java (short-name x)
                                                   shorten (shorten-name (str x))
                                                   :else (str x))
                                      :shape shape, :color color}])]
                :when (seq classes-nodes)]
            (if group-java
              (dot/subgraph (str "cluster_" (safe-name package))
                            (cons {:label package, :color :black} classes-nodes))
              classes-nodes)))))

(defn process-edges [structure {:keys [only-own ignore]}]
  (update-in structure [:edges]
             #(for [[from to] %
                    :when (and (not (ignored? from ignore))
                               (not (ignored? to ignore))
                               (or (not only-own)
                                   (and (.startsWith (str from) @common-ns)
                                        (.startsWith (str to) @common-ns))))]
                [(keyword from) :> (keyword to)])))

(defn depgraph
  "Generate a namespace dependency graph as svg file"
  [project & args]
  (let [args (loop [[key & rst] args, args {}]
               (cond (not key) args
                     (= key "ignore") (assoc args :ignore rst)
                     :else (recur rst (assoc args (keyword key) true))))
        args (assoc args :title (:title project))
        source-path (:source-path project "src")
        dotfile (str (:name project) ".dot")
        svgfile (str (:name project) ".png")]
    (-> source-path
        (parse-directory args)
        ((partial remove (comp nil? first)))
        restructure
        ((fn [x] (reset! common-ns (common-namespace)) x))
        (process-java-nodes "hexagon" args)
        (process-clojure-nodes "ellipse" args)
        (process-edges args)
        (generate-graph args)
        dot/dot
        (dot/save! svgfile {:format :png})
        ;; (doto
        ;;     ((partial spit dotfile))
        ;;   (dot/save! svgfile {:format :svg}))
        )))

#_(depgraph {:name "campus", :source-path "/home/unlogic/work/projects/Java/CampusConvert/src/"} "only-own" "colored")
#_(depgraph {:name "clojure", :source-path "/home/unlogic/clojure/android/clojure/src/jvm/com/"} "only-own" "colored")

#_(depgraph {:name "neko", :source-path "/home/unlogic/clojure/android/neko/src"} "colored" "only-own" )

#_(depgraph {:name "cider-nrepl", :source-path "/tmp/cider-nrepl/src/"} "colored" "only-own" "ignore"
            "cider.nrepl.middleware.util.java.parser" "clojure.lang.*" "java.*")
