(ns allgress.cereus.core
  (:require-macros [freactive.macros :refer [rx non-reactively]])
  (:require [clojure.set]
            [freactive.dom :as dom]))

(enable-console-print!)

;;; TODO - when laziness bug is fixed in freactive, remove this
(defn add-watch
  [iref key f]
  @iref
  @iref
  (cljs.core/add-watch iref key f))

(def bindings (atom {}))

(defn register-property-namespace
  ([namespace]
   (dom/register-attr-prefix!
     namespace
     (fn [element node-state attr-name attr-val]
       (let [prop-name (str namespace "/" attr-name)
             old-val (aget element prop-name)
             attribute-changed (aget element "attributeChangedCallback")]
         (if (= (type attr-val) freactive.core/ReactiveExpression)
           (let [new-val @attr-val]
             (when-not (and (contains? @bindings element) (contains? (@bindings element) prop-name))
               (aset element prop-name new-val)
               (.call attribute-changed element prop-name old-val new-val)
               (add-watch attr-val prop-name
                          (fn [k _ old-val new-val]
                            (aset element prop-name attr-val)
                            (.call attribute-changed element prop-name old-val new-val)))
               (swap! bindings assoc-in [element prop-name] attr-val)))
           (do
             (aset element prop-name attr-val)
             (.call attribute-changed element prop-name old-val attr-val))))))))

(defonce registration-handlers (atom []))

(defonce document-ready (atom true))

(defn- register-element*
  [tag-name register-fn]
  #_(println tag-name "register-element*" @document-ready)
  (if @document-ready
    (register-fn)
    (swap! registration-handlers conj register-fn)))

#_(defn- doc-ready-handler
  []
  (reset! document-ready true)
  (doseq [h @registration-handlers] (h)))

#_(defn- on-doc-ready []
  (.addEventListener js/window "WebComponentsReady" doc-ready-handler))

#_(on-doc-ready)

(let [element-prototypes (aget js/window "element-prototypes")]
  (when (nil? element-prototypes)
    (aset js/window "element-prototypes" (atom {}))))

(let [waiting-for-registration (aget js/window "waiting-for-registration")]
  (when (nil? waiting-for-registration)
    (aset js/window "waiting-for-registration" (atom #{}))))

(defonce reload-listeners (atom #{}))

(defn waiting-for
  []
  (let [waiting-for-registration (aget js/window "waiting-for-registration")]
    (println @waiting-for-registration)))

(defn on-jsload
  []
  (doseq [f @reload-listeners] (f)))

(defn- finalize-registration
  [tag-name js-prototype]
  (let [element-prototypes (aget js/window "element-prototypes")
        waiting-for-registration (aget js/window "waiting-for-registration")]
    (.registerElement js/document (name tag-name) (clj->js {:prototype js-prototype}))
    (swap! waiting-for-registration disj tag-name)
    (swap! element-prototypes assoc tag-name js-prototype))
  #_(println tag-name "REGISTERED BY CEREUS"))

(defn- register-element
  [tag-name js-prototype imports property-namespace]
  (when property-namespace (register-property-namespace property-namespace))
  #_(println tag-name "register-element")
  (let [deps (set (filter keyword? imports))
        element-prototypes (aget js/window "element-prototypes")
        waiting-for-registration (aget js/window "waiting-for-registration")]
    (swap! waiting-for-registration clojure.set/union (clojure.set/difference deps (set (keys @element-prototypes))))
    (if (some? (@element-prototypes tag-name))
      (let [registered-prototype (@element-prototypes tag-name)]
        (aset registered-prototype "createdCallback" (aget js-prototype "createdCallback"))
        (aset registered-prototype "attachedCallback" (aget js-prototype "attachedCallback"))
        (aset registered-prototype "detachedCallback" (aget js-prototype "detachedCallback"))
        (aset registered-prototype "attributeChangedCallback" (aget js-prototype "attributeChangedCallback")))
      (do
        (.import js/Polymer (clj->js (filter string? imports))
                 (fn []
                   #_(println tag-name " IMPORT DONE")
                   #_(println (keys @element-prototypes))
                   (if (empty? (clojure.set/difference deps (set (keys @element-prototypes))))
                     (finalize-registration tag-name js-prototype)
                     (add-watch element-prototypes tag-name
                                (fn [k r o n]
                                  (when-not (some? (@element-prototypes tag-name))
                                    #_(println tag-name " UPDATED")
                                    (let [unregistered-deps (clojure.set/difference deps (set (keys @element-prototypes)))]
                                      (when (empty? unregistered-deps)
                                        (remove-watch element-prototypes tag-name)
                                        (finalize-registration tag-name js-prototype)))))))))))))

(defn keyword->str*
  [x]
  (str (if (namespace x) (str (namespace x) "/")) (name x)))

(def keyword->str (memoize keyword->str*))

(defn map->js
  [x]
  (apply js-obj (mapcat (fn [[k v]] [(keyword->str k) v]) x)))

(defn fire!
  ([el event-type]
   (fire! el event-type {:bubbles true :cancelable true}))
  ([el event-type event-init]
   (let [type (cond
                (keyword? event-type) (name event-type)
                (string? event-type) event-type)
         event-data (js/CustomEvent. type (map->js event-init))]
     (.dispatchEvent el event-data))))

(defn listen!
  [el event-type listener]
  (let [type (cond
               (keyword? event-type) (name event-type)
               (string? event-type) event-type)]
    (.addEventListener el type listener)))

(defn unlisten!
  [el event-type listener]
  (let [type (cond
               (keyword? event-type) (name event-type)
               (string? event-type) event-type)]
    (.removeEventListener el type listener)))

(defn nodelist-to-seq
  "Converts nodelist to (not lazy) seq."
  [nl]
  (let [result-seq (if (array? nl)
                     (map (fn [i] (aget nl i)) (range (.-length nl)))
                     (map #(.item nl %) (range (.-length nl))))]
    (doall result-seq)))

(defn- spec-to-proto
  ([spec tag-name imports]
   (assert (and (some? (:state spec)) (some? (:view spec))) "Custom element spec must contain :state and :view functions")
   (let [scope-name (str (gensym "cereus/scope"))]
     {:createdCallback
      {:value
       (fn []
         #_(println tag-name " CREATED")
         (this-as this
                  (let [reload-listener (atom nil)
                        scope (merge {:this            this
                                      :state           ((:state spec))
                                      :reload-listener reload-listener
                                      :listen-reload   (fn [f]
                                                         (cljs.core/swap! reload-listeners cljs.core/conj f)
                                                         (cljs.core/reset! reload-listener f))}
                                     (if-let [s (:scope spec)]
                                       (if (fn? s) (s) s)
                                       {}))]
                    (aset this scope-name scope)
                    (when-let [created-fn# (:created spec)] (created-fn# (:state scope) this))
                    (when-let [attribute-changed-fn (:attribute-changed spec)]
                      (doseq [attr (nodelist-to-seq (aget this "attributes"))]
                        (attribute-changed-fn (:state scope) this (aget attr "name") nil (aget attr "value")))))))}
      :attachedCallback
      {:value
       (fn []
         (this-as this
                  (let [shadow (.createShadowRoot this)
                        scope (aget this scope-name)]
                    (when-let [attached-fn# (:attached spec)] (attached-fn# (:state scope) this))
                    (dom/mount! shadow ((:view spec) (:state scope) (:this scope)))
                    ((:listen-reload scope) (fn [] (dom/mount! shadow ((:view spec) (:state scope) (:this scope))))))))}
      :detachedCallback
      {:value
       (fn []
         (this-as this
                  (when-let [property-bindings (@bindings this)]
                    (doseq [prop-name (keys property-bindings)]
                      (remove-watch (property-bindings prop-name) prop-name)))
                  (let [scope (aget this scope-name)]
                    (when-let [detached-fn (:detached spec)]
                      (detached-fn (:state scope) this))
                    (when-let [f (:reload-listener scope)]
                      (swap! reload-listeners disj f)))))}
      :attributeChangedCallback
      {:value
       (fn [attr-name old-value new-value]
         (this-as this
                  (let [scope (aget this scope-name)]
                    (when-let [attribute-changed-fn (:attribute-changed spec)]
                      (attribute-changed-fn (:state scope) this attr-name old-value new-value)))))}})))

(defn register-custom-element!
  [tag-name base-element spec & imports]
  (let [js-prototype (.create js/Object (.-prototype base-element) (clj->js (spec-to-proto spec tag-name imports)))]
    (register-element* tag-name #(register-element tag-name js-prototype imports (:property-namespace spec)))))

