(ns sparkplug.listener
  "Functions for creating and working with Spark listeners."
  (:require
    [sparkplug.scala :as scala])
  (:import
    java.util.Properties
    org.apache.spark.api.java.JavaSparkContext
    (org.apache.spark.scheduler
      StageInfo
      SparkListenerJobStart)
    org.apache.spark.SparkContext
    sparkplug.listener.ListenerBridge))


(defmulti parse-event
  "Parses an incoming Spark listener event into a clojure map"
  class)


(defmethod parse-event :default [event] nil)


(defn- parse-stage-info
  [^StageInfo stage-info]
  (let [submission-time (-> stage-info .submissionTime scala/resolve-option)
        completion-time (-> stage-info .completionTime scala/resolve-option)
        failure-reason (-> stage-info .failureReason scala/resolve-option)]
    (merge
      {:stage-id (.stageId stage-info)
       :name (.name stage-info)
       :num-tasks (.numTasks stage-info)
       :details (.details stage-info)
       :status (.getStatusString stage-info)}
      (when submission-time {:submission-time submission-time})
      (when completion-time {:completion-time completion-time})
      (when failure-reason {:failure-reason failure-reason}))))


(defn- parse-properties
  [^Properties prop]
  (reduce (fn [m [k v]] (assoc m k v)) {} prop))


(defmethod parse-event SparkListenerJobStart
  [^SparkListenerJobStart event]
  {:type :job-start
   :job-id (.jobId event)
   :time (.time event)
   :properties (parse-properties (.properties event))})


(defprotocol Listener
  (on-event [this event]))


(defn register
  "Registers the given listener to the given spark context. Returns the bridge object
  that was constructed, which is needed to unregister the listener."
  [^JavaSparkContext context listener]
  (let [^SparkContext sc (.sc context)
        listener-fn (fn [event]
                      (when-let [parsed-event (parse-event event)]
                        (on-event listener parsed-event)))
        bridge (ListenerBridge. listener-fn)]
    (.addSparkListener sc bridge)
    bridge))


(defn unregister
  "Unregisters the given lister bridge object from the given spark context."
  [^JavaSparkContext context bridge]
  (let [^SparkContext sc (.sc context)]
    (.removeSparkListener sc bridge)))
