(ns antistock.config.core
  (:require [antistock.config.schema :refer :all]
            [schema.core :as s]
            [clojure.edn :as edn]))

(s/defn api :- Api
  "Return the API config from `env`."
  [env :- Env]
  {:bind-address (:as-api-bind-address env)
   :bind-port (or (edn/read-string (:as-api-bind-port env))
                  (edn/read-string (:port env)))})

(s/defn api-client :- ApiClient
  "Return the API client config from `env`."
  [env :- Env]
  {:scheme (keyword (:as-api-scheme env))
   :server-name (:as-api-server-name env)
   :server-port (edn/read-string (:as-api-server-port env))})

(s/defn aws :- Aws
  "Return the AWS config from `env`."
  [env :- Env]
  {:access-key (:as-aws-access-key env)
   :account-number (:as-aws-account-number env)
   :secret-key (:as-aws-secret-key env)})

(s/defn broker :- Broker
  "Return the broker config from `env`."
  [env :- Env]
  {:password (:as-broker-password env)
   :scheme (or (:as-broker-scheme env) :amqp)
   :server-name (:as-broker-server-name env)
   :server-port (edn/read-string (:as-broker-server-port env))
   :username (:as-broker-username env)
   :vhost (or (:as-broker-vhost env) "/")})

(s/defn curator :- Curator
  "Return the Curator config from `env`."
  [env :- Env]
  {:password (:as-zookeeper-password env)
   :server-name (:as-zookeeper-server-name env)
   :server-port (edn/read-string (:as-zookeeper-server-port env))
   :username (:as-zookeeper-username env)})

(s/defn db :- Db
  "Return the database config from `env`."
  [env :- Env]
  {:name (:as-db-name env)
   :password (:as-db-password env)
   :pool (some-> env :as-db-pool keyword)
   :test? (true? (edn/read-string (:as-db-test env)))
   :scheme (keyword (:as-db-scheme env))
   :server-name (:as-db-server-name env)
   :server-port (edn/read-string (:as-db-server-port env))
   :username (:as-db-username env)})

(s/defn twitter :- Twitter
  "Return the Twitter config from `env`."
  [env :- Env]
  {:consumer-key (:as-twitter-consumer-key env)
   :consumer-secret (:as-twitter-consumer-secret env)
   :access-token (:as-twitter-access-token env)
   :access-token-secret (:as-twitter-access-token-secret env)})

(s/defn twitter-worker :- TwitterWorker
  "Return the config for the Twitter worker from `env`."
  [env :- Env]
  {:broker (broker env)
   :exchange "tweetlogd"
   :prefetch-count 1
   :queue "worker.tweets"
   :routing-key "twitter.status"})

(s/defn reload :- Reload
  "Return the namespace reload config from `env`."
  [env :- Env]
  {:disabled (true? (edn/read-string (:as-reload-disabled env)))
   :source-paths ["src" "test"]})

(s/defn spark :- Spark
  "Return the Spark application config."
  [env :- Env]
  {:app-name (or (:as-spark-app-name env) "antistock")
   :master (or (:as-spark-master env) "local")})

(s/defn web :- Web
  "Return the web config from `env`."
  [env :- Env]
  {:bind-address (:as-web-bind-address env)
   :bind-port (edn/read-string (:as-web-bind-port env))})

(s/defn web-client :- WebClient
  "Return the web client config from `env`."
  [env :- Env]
  {:scheme (keyword (:as-web-scheme env))
   :server-name (:as-web-server-name env)
   :server-port (edn/read-string (:as-web-server-port env))})

(s/defn wikipedia-views-worker :- WikipediaViewsWorker
  "Return the config for the Wikipedia page views worker from `env`."
  [env :- Env]
  {:broker (broker env)
   :exchange "wikipedia"
   :prefetch-count 1
   :queue "worker.wikipedia.views"
   :routing-key "wikipedia.page.views"})

(s/defn zookeeper :- Zookeeper
  "Return the Zookeeper config from `env`."
  [env :- Env]
  (curator env))

(s/defn api-system :- ApiSystem
  "Return the API system config from `env`."
  [env :- Env]
  {:api (api env)
   :api-client (api-client env)
   :broker (broker env)
   :db (db env)})

(s/defn tweetlogd-system :- TweetlogdSystem
  "Return the worker system config from `env`."
  [env :- Env]
  {:broker (broker env)
   :curator (curator env)
   :db (db env)})

(s/defn worker-system :- WorkerSystem
  "Return the worker system config from `env`."
  [env :- Env]
  {:broker (broker env)
   :db (db env)
   :twitter (twitter-worker env)
   :wikipedia-views (wikipedia-views-worker env)})

(s/defn config :- Config
  "Return the system config from `env`."
  [env :- Env]
  {:api (api env)
   :api-client (api-client env)
   :aws (aws env)
   :broker (broker env)
   :curator (curator env)
   :db (db env)
   :reload (reload env)
   :twitter (twitter env)
   :web (web env)
   :web-client (web-client env)
   :zookeeper (zookeeper env)})
