(ns antistock.queue
  (:refer-clojure :exclude [replace])
  (:require [antistock.system :as system]
            [clojure.edn :as edn]
            [clojure.string :refer [replace]]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [environ.core :refer [env]]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.consumers :as consumers]
            [langohr.core :as core]
            [langohr.exchange :as ex]
            [langohr.queue :as queue]
            [no.en.core :as noencore]))

(def declare-queue queue/declare)

(def publish basic/publish)

(defn publish-edn [channel exchange routing-key payload & [opts]]
  (publish channel exchange routing-key (pr-str payload)
           (merge {:content-type "application/edn"} opts)))

(defn read-edn-payload [payload]
  (edn/read-string {:readers *data-readers*} (String. payload "UTF-8")))

(defn subscribe-edn [channel queue f & [opts]]
  (consumers/subscribe
   channel queue
   (fn [channel metadata payload]
     (f channel metadata (read-edn-payload payload)))
   opts))

(defn open-channel [broker]
  (channel/open (:connection broker)))

(defmacro with-channel [[channel-sym broker] & body]
  `(with-open [~channel-sym (channel/open (:connection ~broker))]
     ~@body))

(defn declare-topology [broker]
  (with-channel [channel broker]
    (ex/declare channel "api.tweets" "fanout"
                {:auto-delete false :durable true})
    (doseq [queue ["flume.tweets" "worker.tweets"]]
      (queue/declare channel queue {:auto-delete false :durable true})
      (queue/bind channel queue "api.tweets"))))

(defn connect [broker]
  (core/connect
   {:automatically-recover true
    :automatically-recover-topology true
    :host (:server-name broker)
    :port (:server-port broker)
    :username (:username broker)
    :password (:password broker)
    :vhost (:vhost broker)}))

(defrecord RabbitMQ [connection server-name server-port username password vhost]
  component/Lifecycle
  (start [component]
    (if (:connection component)
      component
      (let [connection (connect component)
            component (assoc component :connection connection)]
        (declare-topology component)
        (log/infof (str "Connection to RabbitMQ vhost %s on %s, port %d "
                        "as user %s established.")
                   vhost server-name server-port username)
        component)))
  (stop [component]
    (when-let [connection (:connection component)]
      (.close connection)
      (log/infof (str "Connection to RabbitMQ vhost %s on %s, port %d "
                      "as user %s closed.")
                 vhost server-name server-port username))
    (assoc component :connection nil)))

(defn new-broker
  "Returns the message broker component."
  [config] (map->RabbitMQ config))

(defmacro with-broker
  "Eval `body` within a broker context."
  [[broker-sym config] & body]
  `(system/with-component [~broker-sym (new-broker ~config)]
     ~@body))
