(ns antistock.time
  (:refer-clojure :exclude [extend second])
  (:require [antistock.util :refer [immigrate]]
            [clj-time.core :refer :all]
            [clj-time.format :refer :all]
            [clj-time.coerce :refer [to-date-time to-date]]
            [clj-time.periodic :refer [periodic-seq]])
  (:import java.io.Writer
           org.joda.time.DateTime))

(def ^:dynamic *time-formatter* :date-time-no-ms)

(defn hourly-time-series [& [start end]]
  (let [start (or (to-date-time start) (minus (now) (hours 2)))
        end (or (to-date-time end) (plus start (hours 1)))]
    (map #(plus start (hours %1))
         (take (/ (- (.getMillis end)
                     (.getMillis start))
                  (* 1000 60 60))
               (iterate inc 0)))))

(defn truncate-time
  "Set the minutes, seconds and millis of `time` to zero."
  [time where]
  (case where
    :minutes (-> (to-date-time time)
                 (.withMinuteOfHour 0)
                 (.withSecondOfMinute 0)
                 (.withMillisOfSecond 0))))

(defn parse-wikipedia-time
  "Parse `time` using the pattern HH:mm, dd MMM YYYY."
  [time]
  (try (parse (formatter "HH:mm, dd MMM YYYY") (str time))
       (catch IllegalArgumentException e nil)))

(defmethod print-dup DateTime
  [^DateTime d ^Writer w]
  (print-dup (.toDate d) w))

(defmethod print-method DateTime
  [^DateTime d ^Writer w]
  (print-method  (.toDate d) w))

(defn ^String format-time
  "Format the object with the default or the given time formatter."
  [object & [formatter]]
  (if object
    (unparse (or formatter (formatters *time-formatter*))
             (to-date-time object))))

(defn year-month-day
  "Format `time` in YYYY-MM-DD format."
  [time]
  (if-let [time (to-date-time time)]
    (unparse (formatters :year-month-day) time )))

(defn time-range
  "Return a lazy sequence of DateTime's from start to end, incremented
  by 'step' units of time."
  [start end step]
  (if (before? end start)
    (reverse (time-range end start step))
    (let [inf-range (periodic-seq start step)
          below-end? (fn [t] (within? (interval start end) t))]
      (take-while below-end? inf-range))))

(defn monthly-date-range
  "Return a lazy sequence of dates from `start` to `end`,
  incremented by an monthly interval."
  [start end]
  (let [start (or (to-date-time start) (now))
        end (or (to-date-time end) (plus start (days 1)))]
    (map to-date (time-range start end (months 1)))))
