(ns antistock.transit
  (:require [clj-http.client]
            [cognitect.transit :as transit]
            [geo.transit :as geo-transit])
  (:import (java.io ByteArrayInputStream ByteArrayOutputStream)))

(def read-handlers
  geo-transit/read-handlers)

(def write-handlers
  geo-transit/write-handlers)

(defn encode
  "Encode `payload` into Transit format."
  [payload & [opts]]
  (let [format (or (:format opts) :json)
        stream (ByteArrayOutputStream.)
        writer (transit/writer stream format {:handlers write-handlers})
        _ (transit/write writer payload)]
    (case format
      :json (.toString stream)
      :msgpack (.toByteArray stream))))

(defn decode
  "Decode `payload` from Transit format."
  [payload & [opts]]
  (let [format (or (:format opts) :json)
        stream (ByteArrayInputStream.
                (case format
                  :json (.getBytes payload)
                  :msgpack payload))
        reader (transit/reader stream format {:handlers read-handlers})]
    (transit/read reader)))

(defn coerce-transit-body
  [request {:keys [body] :as resp}]
  (assoc resp :body (decode (slurp body))))

(defmethod clj-http.client/coerce-content-type :application/transit+json
  [req resp]
  (coerce-transit-body req resp))

(defn wrap-transit-form-params
  "Middleware wrapping the submission of form parameters in the transit fromat."
  [client]
  (fn [{:keys [form-params content-type method request-method]
        :or {content-type :x-www-form-urlencoded}
        :as req}]
    (if (and form-params
             (#{:post :put :patch} (or request-method method))
             (#{:transit
                :application/transit+json
                "application/transit+json"}
              content-type))
      (client (-> req
                  (dissoc :form-params)
                  (assoc :content-type "application/transit+json"
                         :body (encode form-params))))
      (client req))))
