(ns antistock.util
  (:refer-clojure :exclude [replace])
  (:require [clj-http.client :as http]
            [clj-time.coerce :refer [to-date-time]]
            [clj-time.format :refer [formatter parse]]
            [clojure.data.csv :refer [read-csv]]
            [clojure.java.io :refer [delete-file file reader resource]]
            [clojure.string :refer [blank? replace split]]
            [clojure.tools.logging :as log]
            [inflections.core :refer [hyphenate]]
            [no.en.core :refer [parse-double parse-integer]]
            [clojure.edn :as edn])
  (:import [java.io BufferedReader StringReader]
           java.util.UUID))

(def ^:dynamic *yahoo-date-formatter*
  (formatter "MM/dd/yyyy"))

(defn slurp-edn
  "Slurp `filename` in EDN format."
  [filename]
  (->> (slurp filename)
       (edn/read-string {:readers *data-readers*})))

(defn ask
  "When `answer` is nil ask the user a `question` and return the
  result, otherwise return `answer`."
  [question & [answer]]
  (if answer
    answer
    (do (print question)
        (.flush *out*)
        (read-line))))

(defn banner
  "Returns the Antistock banner."
  [] (slurp (resource "banner.txt")))

(defn log-banner
  "Log the Antistock banner."
  [& [name]]
  (doseq [line (split (banner) #"\n")]
    (log/info line))
  (log/info "")
  (when name
    (let [padding (- (/ (- 60 (count name)) 2) 4)]
      (log/infof "%s⋆⋆⋆ %s ⋆⋆⋆"
                 (apply str (repeat padding " "))
                 name)
      (log/info ""))))

(defn beginning-of-year
  "Returns the beginning of the year for `time`."
  [time]
  (-> (.withMonthOfYear (to-date-time time) 1)
      (.withDayOfMonth 1)))

(defn csv-seq->map
  "Convert a seq of vectors into a seq of maps."
  [s & {:keys [keys]}]
  (let [ks (if keys keys (first s))
        ks (map (comp keyword hyphenate) ks)]
    (map #(zipmap ks %1)
         (if keys s (rest s)))))

(defn crc-path
  "Returns the path of the Wikipedia page view dump at `time`."
  [path]
  (if-not (blank? (str path))
    (let [file (file path)]
      (replace
       (str file)
       (.getName file)
       (str "." (.getName file) ".crc")))))

(defn immigrate
  "Create a public var in this namespace for each public var in the
  namespaces named by ns-names. The created vars have the same name, root
  binding, and metadata as the original except that their :ns metadata
  value is this namespace."
  [& ns-names]
  (doseq [ns ns-names]
    (require ns)
    (doseq [[sym var] (ns-publics ns)]
      (let [v (if (.hasRoot var)
                (var-get var))
            var-obj (if v (intern *ns* sym v))]
        (when var-obj
          (alter-meta! var-obj
                       (fn [old] (merge (meta var) old)))
          var-obj)))))

(defn indexed
  "Returns a lazy sequence of [index, item] pairs, where items come
  from 's' and indexes count up from zero.

  (indexed '(a b c d))  =>  ([0 a] [1 b] [2 c] [3 d])"
  [s]
  (map vector (iterate inc 0) s))

(defn print-banner [name]
  (println (slurp (resource (format "banner/%s.txt" name)))))

(defn read-lines
  "Like clojure.core/line-seq but opens f with reader.  Automatically
  closes the reader AFTER YOU CONSUME THE ENTIRE SEQUENCE."
  [f]
  (let [read-line (fn this [^BufferedReader rdr]
                    (lazy-seq
                     (if-let [line (.readLine rdr)]
                       (cons line (this rdr))
                       (.close rdr))))]
    (read-line (reader f))))

(defn wrap-csv-response
  "Returns a HTTP client that parses :body into a seq of vectors if
  the content-type header matches text/csv."
  [client]
  (fn [request]
    (let [response (client request)
          content-type (get (:headers response) "content-type")]
      (if (and (string? content-type)
               (re-matches #"(?i)text/csv" content-type))
        (do
          (update-in
           response [:body]
           (fn [body]
             (with-open [reader (BufferedReader. (StringReader. body))]
               (doall (read-csv reader))))))
        response))))

(def http
  (-> http/request
      wrap-csv-response))

(defn parse-yahoo-date
  "Parse `s` as Yahoo's mm/dd/yyyy formatted date."
  [s] (try (if s (parse *yahoo-date-formatter* s))
           (catch IllegalArgumentException _ nil)))

(defn remove-nil
  "Remove all entries from `m` where value is nil."
  [m]
  (->> (for [[k v] m :when (nil? v)] k)
       (apply dissoc m)))

(defn delete-files
  "Delete all filenames."
  [filenames & [silently]]
  (doseq [filename filenames]
    (delete-file (file filename) silently)))

(defmacro with-tmp-files
  "Evaluate `body` with `tmp-syms` to temporary file names, that are
  delete after `body` has been evaluated."
  [[& tmp-syms] & body]
  (let [tmp-paths (mapcat #(vector %1 `(str "/tmp/" (UUID/randomUUID)))
                          tmp-syms)]
    `(let [~@tmp-paths]
       (try
         ~@body
         (finally
           (delete-files [~@tmp-syms] true))))))
