(ns antistock.curator.core
  (:require [antistock.zookeeper.core :as zookeeper]
            [clojure.edn :as edn]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [schema.core :as s]
            [zookeeper :as zk]
            [zookeeper.internal :as zi])
  (:import org.apache.curator.ensemble.fixed.FixedEnsembleProvider
           [org.apache.curator.framework
            AuthInfo CuratorFramework CuratorFrameworkFactory]
           org.apache.curator.framework.api.ACLProvider
           org.apache.curator.retry.ExponentialBackoffRetry
           org.apache.zookeeper.data.ACL))

(def ^:dynamic *defaults*
  "The default Curator configuration."
  {:data-readers *data-readers*
   :default-perms zk/default-perms
   :server-name "localhost"
   :server-port 2181
   :username nil
   :password nil})

(s/defrecord Curator
    [connection :- (s/maybe CuratorFramework)
     data-readers :- {s/Any s/Any}
     server-name :- String
     server-port :- s/Int
     username :- (s/maybe String)
     password :- (s/maybe String)]
  {s/Any s/Any})

(defn import-classes
  "Import common curator classes."
  []
  (import '[org.apache.curator.framework.recipes.cache
            PathChildrenCache PathChildrenCacheListener
            PathChildrenCacheEvent PathChildrenCacheEvent$Type]
          '[org.apache.curator.framework.recipes.nodes
            GroupMember]
          '[org.apache.curator.framework.recipes.leader
            LeaderLatch LeaderLatchListener LeaderLatch$CloseMode]))

(s/defn connection-string :- String
  "Return the connection string for `curator`."
  [curator :- Curator]
  (str (:server-name curator)
       (some->> curator :server-port (str ":"))))

(s/defn default-acl :- ACL
  "Return the default ACL of `curator`."
  [{:keys [username password default-perms]} :- Curator]
  (if username
    (apply zookeeper/digest-acl username password default-perms)
    (apply zk/world-acl default-perms)))

(s/defn default-acl-provider :- (s/maybe ACLProvider)
  "Return the digest auth info `curator`."
  [{:keys [username password default-perms]} :- Curator]
  (let [acl (if username
              (apply zookeeper/digest-acl username password default-perms)
              (apply zk/world-acl default-perms))]
    (reify ACLProvider
      (getAclForPath [this path]
        (list acl))
      (getDefaultAcl [this]
        (list acl)))))

(s/defn ^String generate-digest :- (s/maybe String)
  "Generate the digest for `curator`."
  [curator :- Curator]
  (when (:username curator)
    (str (:username curator) ":" (:password curator))))

(s/defn digest-auth-info :- (s/maybe AuthInfo)
  "Return the digest auth info `curator`."
  [curator :- Curator]
  (when (:username curator)
    (AuthInfo. "digest" (.getBytes (generate-digest curator) "UTF-8"))))

(s/defn curator-framework :- CuratorFramework
  "Return the connection string for `curator`."
  [curator :- Curator]
  (let [ensemble (FixedEnsembleProvider. (connection-string curator))]
    (.build (cond-> (doto (CuratorFrameworkFactory/builder)
                      (.ensembleProvider ensemble)
                      (.retryPolicy (ExponentialBackoffRetry. 1000 10)))
              (:username curator)
              (.aclProvider (default-acl-provider curator))
              (:username curator)
              (.authorization (list (digest-auth-info curator)))
              (:connection-timeout-ms curator)
              (.connectionTimeoutMs (:connection-timeout-ms curator))
              (:session-timeout-ms curator)
              (.sessionTimeoutMs (:session-timeout-ms curator))))))

(s/defn connect :- Curator
  "Connect `curator` to ZooKeeper."
  [curator :- Curator]
  (->> (doto (curator-framework curator) (.start))
       (assoc curator :connection)))

(s/defn ^CuratorFramework connection :- (s/maybe CuratorFramework)
  "Return the curator connection."
  [curator :- Curator]
  (:connection curator))

(s/defn connected? :- s/Bool
  "Returns true if `curator` is connected, otherwise false."
  [curator :- Curator]
  (instance? CuratorFramework (connection curator)))

(s/defn disconnect :- Curator
  "Disconnect `curator` from ZooKeeper."
  [curator :- Curator]
  (some-> curator connection .close)
  (assoc curator :connection nil))

(s/defn encode :- s/Any
  "Encode `data` into bytes via `pr-str`."
  [curator :- Curator data :- s/Any]
  (.getBytes (pr-str data)))

(s/defn decode :- s/Any
  "Decode the bytes in `data` using :data-readers from `curator`."
  [{:keys [data-readers] :as curator} :- Curator data :- s/Any]
  (edn/read-string {:data-readers data-readers} (String. ^bytes data "UTF-8")))

(s/defn acl :- s/Any
  "Get the ACL for node at `path` in Zookeeper using the `curator` client."
  [curator :- Curator path :- String & [data :- s/Any opts]]
  (-> (.getACL (connection curator))
      (.forPath path)))

(defn- create-mode
  [{:keys [persistent? sequential?]
    :or {persistent? false
         sequential? false}}]
  (zi/create-modes {:persistent? persistent? :sequential? sequential?}))

(s/defn create :- s/Any
  "Create the node at `path` in Zookeeper using the `curator` client."
  [curator :- Curator path :- String & [data :- s/Any opts]]
  (-> (.create (connection curator))
      (.withMode (create-mode opts))
      (.forPath path (encode curator data))))

(s/defn delete :- s/Any
  "Delete the node at `path` in Zookeeper."
  [curator :- Curator path :- String & [opts]]
  {:pre [(connected? curator)]}
  (.forPath (.delete (connection curator)) path))

(s/defn data :- s/Any
  "Return the data at `path` from the Zookeeper node using the
  `curator` client."
  [curator :- Curator path :- String & [opts]]
  {:pre [(connected? curator)]}
  (decode curator (.forPath (.getData (connection curator)) path)))

(s/defn exists? :- s/Bool
  "Returns true if the node at `path` exists in Zookeeper, otherwise false.
  client."
  [curator :- Curator path :- String]
  {:pre [(connected? curator)]}
  (some? (.forPath (.checkExists (connection curator)) path)))

(s/defn set-data :- s/Any
  "Set data at `path` to the Zookeeper node using the `curator`
  client."
  [curator :- Curator path :- String data :- s/Any & [opts]]
  {:pre [(connected? curator)]}
  (.forPath (.setData (connection curator)) path (encode curator data)))

(s/defn curator :- Curator
  "Return a new Curator framework."
  [& [config]]
  (map->Curator (merge *defaults* config)))

(extend-protocol component/Lifecycle
  Curator
  (start [curator]
    (if (:connection curator)
      curator
      (let [curator (connect curator)]
        (log/info "Curator framework started.")
        curator)))
  (stop [curator]
    (let [curator (disconnect curator)]
      (log/info "Curator framework stopped.")
      curator)))

(defmacro with-curator
  "Create a new curator, bind the started curator to
  `curator-sym`, evaluate `body` and stop the curator again."
  [[curator-sym config] & body]
  `(let [curator# (component/start (curator ~config))
         ~curator-sym curator#]
     (try ~@body
          (finally (component/stop curator#)))))
