(ns antistock.db.companies
  (:refer-clojure :exclude [distinct group-by update])
  (:require [antistock.db.util :refer [fulltext]]
            [clj-time.coerce :refer [to-date-time]]
            [clojure.java.jdbc :as jdbc]
            [datumbazo.core :refer :all]))

(deftable companies
  "The companies database table."
  (column :id :serial :primary-key? true)
  (column :country-id :integer :references :countries/id)
  (column :industry-id :integer :references :industries/id)
  (column :wikipedia-page-id :integer :references :wikipedia.pages/id)
  (column :name :citext :not-null? true :unique? true)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(defquery companies
  "Returns the companies."
  [db & [opts]]
  (select db [:companies.*]
    (from :companies)
    (fulltext (:query opts) :companies.name)
    (paginate (:page opts) (:per-page opts))
    (order-by :name)))

(defquery companies-by-updated-at-asc
  "Returns the companies by updated-at column in asc order."
  [db] (compose (companies* db)
                (order-by (asc :updated-at))))

(defn all
  "Return the companies."
  [db & [opts]]
  @(select db [:companies.id
               :companies.name
               :companies.created-at
               :companies.updated-at
               (as '(json_build_object
                     "id" :countries.id
                     "name" :countries.name)
                   :country)
               (as '(json_build_object
                     "id" :industries.id
                     "name" :industries.name)
                   :industry)
               (as '(json_build_object
                     "id" :sectors.id
                     "name" :sectors.name)
                   :sector)
               (as '(json_build_object
                     "id" :sectors.id
                     "name" :sectors.name)
                   :wikipedia)]
     (from :companies)
     (join :countries.id :companies.country-id)
     (join :industries.id :companies.industry-id)
     (join :sectors.id :companies.sector-id)
     (fulltext (:query opts) :companies.name)
     (paginate (:page opts) (:per-page opts))
     (order-by :companies.name)))
