(ns antistock.db.countries
  (:refer-clojure :exclude [distinct group-by update])
  (:require [antistock.db.continents :as continents]
            [clojure.java.jdbc :as jdbc]
            [clojure.tools.logging :refer [errorf infof]]
            [clojure.set :refer [rename-keys]]
            [datumbazo.core :refer :all]
            [geonames.countries :as geonames]))

(deftable countries
  "The countries database table."
  (column :id :serial :primary-key? true)
  (column :continent-id :integer :not-null? true :references :continents/id)
  (column :name :citext :not-null? true :unique? true)
  (column :iso-3166-1-alpha-2 :citext :not-null? true :unique? true)
  (column :iso-3166-1-alpha-3 :citext :not-null? true :unique? true)
  (column :iso-3166-1-numeric :integer :not-null? true :unique? true)
  (column :fips-code :citext)
  (column :geonames-id :integer :not-null? true :unique? true)
  (column :phone-prefix :citext)
  (column :area :biginteger)
  (column :population :biginteger)
  (column :geom :geometry :hidden? true)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(defn germany
  "Find Germany."
  [db] (country-by-iso-3166-1-alpha-2 db "de"))

(defn indonesia
  "Find Indonesia."
  [db] (country-by-iso-3166-1-alpha-2 db "id"))

(defn spain
  "Find Spain."
  [db] (country-by-iso-3166-1-alpha-2 db "es"))

(defn united-states
  "Find the United States."
  [db] (country-by-iso-3166-1-alpha-2 db "us"))

(defn find-country
  "Find a country by one of it's unique attributes."
  [db country]
  (or (country-by-id db (:id country))
      (country-by-geonames-id db (:geonames-id country))
      (country-by-iso-3166-1-alpha-2 db (:iso-3166-1-alpha-2 country))
      (country-by-iso-3166-1-alpha-3 db (:iso-3166-1-alpha-3 country))
      (country-by-iso-3166-1-numeric db (:iso-3166-1-numeric country))
      (country-by-fips-code db (:fips-code country))))

(defn load-countries-from-geonames
  "Load all countries from geonames.org into the database."
  [db]
  (jdbc/with-db-transaction
    [db db]
    (->> (for [candidate (map #(rename-keys %1 {:id :geonames-id}) (geonames/countries))]
           (let [country (find-country db candidate)
                 savepoint (.setSavepoint (:connection db))]
             (try (if country
                    (update-country db (merge country candidate))
                    (if-let [continent (continents/continent-by-code db (:continent-code candidate))]
                      (insert-country db (assoc  candidate :continent-id (:id continent)))
                      (errorf "Cant't find continent for country %s by code %s." (:name candidate) (:continent-code candidate))))
                  (catch Exception e
                    (.rollback (:connection db) savepoint)
                    (errorf "Cant't save country %s: %s." (:name candidate) (.getMessage e))
                    (errorf "%s" (pr-str candidate))))))
         (remove nil?)
         (doall))))

(defn all
  "Return all countries."
  [db & [opts]]
  @(select db [:countries.id
               :countries.name
               :countries.created-at
               :countries.updated-at
               :countries.iso-3166-1-alpha-2
               :countries.iso-3166-1-alpha-3
               :countries.iso-3166-1-numeric
               (as '(json_build_object
                     "id" :continents.id
                     "name" :continents.name)
                   :continent)]
     (from :countries)
     (join :continents.id :countries.continent-id)
     (paginate (:page opts) (:per-page opts))))
