(ns antistock.db.util
  (:refer-clojure :exclude [distinct group-by update])
  (:require [antistock.time :refer [current-duration-str]]
            [clojure.string :refer [blank?]]
            [datumbazo.core :refer :all]
            [sqlingvo.util :refer [sql-name]]
            [clj-time.core :refer [now]]
            [clj-time.coerce :refer [to-sql-time]]
            [clojure.string :as str]
            [clojure.tools.logging :as log])
  (:import [java.sql Timestamp]))

(defn column-keyword [table column]
  (keyword (str (name table) "." (name column))))

(defn json-embed-object
  "Return the SQL expression for a JSON embedded country."
  [alias table primary-key columns]
  (as `(case (is-null ~(column-keyword table primary-key)) nil
             (json_build_object
              ~@(mapcat (fn [column]
                          [(name column) (column-keyword table column)])
                        columns)))
      alias))

(defn json-embed-country
  "Return the SQL expression for a JSON embedded country."
  []
  (json-embed-object :country :countries :id [:id :name]))

(defn json-embed-industry
  "Return the SQL expression for a JSON embedded industry."
  []
  (json-embed-object :industry :industries :id [:id :name]))

(defn json-embed-sector
  "Return the SQL expression for a JSON embedded sector."
  []
  (json-embed-object :sector :sectors :id [:id :name]))

(defn fulltext
  "Add a where condition to a select query."
  [query & columns]
  (when-not (blank? query)
    (where `(or (~(keyword "@@")
                 (to_tsvector (array_to_string [~@columns], " "))
                 (plainto_tsquery ~query))
                ~@(map #(list 'like %1 (str "%" query "%")) columns))
           :and)))

(defn generate-date-series
  "Generate a date series from `start` to `end`."
  [db start end]
  (select db [(as '(cast (date_trunc "day" :time) :date) :date)]
    (from (as `(generate_series
                (cast ~(to-sql-time start) :timestamp)
                (cast ~(to-sql-time end) :timestamp)
                (cast "1 day" :interval))
              :time))))

(defn generate-time-ranges
  "Generate time ranges between `start` and `end` with each range
  having an interval of `size`."
  [db start end size]
  (select db [(as :timestamp :start)
              (as `(+ :timestamp (cast ~size :interval)) :end)
              (as `(tstzrange
                    :timestamp
                    (+ :timestamp (cast ~size :interval)))
                  :range)]
    (from (as `(generate_series
                (cast ~start :timestamp)
                (cast ~end :timestamp)
                (cast ~size :interval))
              :timestamp))
    (where `(< :timestamp ~end))))

(defn materialized-views
  "Return all materialized views"
  [db]
  (->> @(select db [(as '(cast oid :regclass) :name)]
          (from :pg_class)
          (where `(= :pg_class.relkind "m")))
       (map (comp keyword :name))))

(defn refresh-materialized-views
  "Refresh all materialized views."
  [db]
  (let [views [:ml.uflw :ml.sentpos :ml.tgeo :ml.tuday :ml.sentneg
               :ml.tid :ml.thtg :ml.uid :ml.tday :ml.sentneut :ml.rtu :ml.ufrn
               :ml.turl :ml.rtid :ml.sentvneg :ml.tusm :ml.sentvpos
               :ml.features]]
    (doseq [view views]
      (let [started-at (now)]
        @(refresh-materialized-view db view)
        (log/infof "Refreshed materialized view %s in %s."
                   (sql-name db view)
                   (current-duration-str started-at))))
    views))
