(ns antistock.db.quotes
  (:refer-clojure :exclude [distinct group-by update])
  (:require [antistock.db.util :refer [fulltext]]
            [clj-time.coerce :refer [to-date-time]]
            [clojure.string :refer [lower-case]]
            [datumbazo.core :refer :all]))

(deftable quotes
  "The quotes database table."
  (column :id :serial :primary-key? true)
  (column :exchange-id :integer :not-null? true :references :exchanged/id)
  (column :company-id :integer :references :companies/id)
  (column :symbol :citext :not-null? true :unique? true)
  (column :created-at :timestamp-with-time-zone :not-null? true :default "now()")
  (column :updated-at :timestamp-with-time-zone :not-null? true :default "now()"))

(defquery quotes
  "Returns all quotes."
  [db & [opts]]
  (select db [:quotes.*]
    (from :quotes)
    (fulltext (:query opts) :quotes.symbol)
    (paginate (:page opts) (:per-page opts))))

(defquery quotes-mentioned
  "Returns all quotes mentioned in `text`."
  [db text]
  (select db [:quotes.*]
    (from :quotes)
    (join :companies '(on (= :companies.id :quotes.company-id)))
    (where `(or (~(keyword "~") ~text (concat "(^|\\s)\\$" :quotes.symbol "($|\\s)"))
                (like ~text (concat "%" :companies.name "%"))))))

(defquery by-symbols
  "Return all quotes for `symbols`."
  [db symbols]
  (select db [:quotes.*]
    (from :quotes)
    (join :companies '(on (= :companies.id :quotes.company-id)))
    (where `(in (lower :quotes.symbol)
                ~(apply list (map lower-case symbols))))))

(defquery random-quotes
  "Return `n` random quotes."
  [db n]
  (select db [*]
    (from :quotes)
    (order-by '(random))
    (limit n)))

(defquery quotes-with-pos-daily-return
  "Return quotes that have a positive daily return at `date`."
  [db date & {:keys [page per-page]}]
  (select db [:quotes.* :daily-return]
    (from :quotes)
    (join :prices '(on (= :prices.quote-id :quotes.id)))
    (where `(and (> :daily-return 0)
                 (= :date ~(to-date-time date))))
    (order-by (desc :daily-return))
    (paginate page per-page)))

(defn wikipedia-pages
  "Return the Wikipedia pages for `quote`."
  [db quote]
  (select db [:wikipedia.pages.*]
    (from :wikipedia.pages)
    (join :companies '(on (= :companies.wikipedia-page-id :pages.id)))
    (join :quotes '(on (= :quotes.company-id :companies.id)))
    (where `(= :quotes.id ~(:id quote)))))

(defn select-quotes
  "Return the quotes."
  [db & [opts]]
  (select db [:quotes.id
              :quotes.symbol
              :quotes.created-at
              :quotes.updated-at
              (as '(json_build_object
                    "id" :companies.id
                    "name" :companies.name)
                  :company)
              (as '(json_build_object
                    "id" :exchanges.id
                    "name" :exchanges.name)
                  :exchange)]
    (from :quotes)
    (join :companies.id :quotes.company-id)
    (join :exchanges.id :quotes.exchange-id)
    (fulltext (:query opts) :quotes.symbol)
    (paginate (:page opts) (:per-page opts))
    (order-by :quotes.symbol)))

(defn all
  "Return the quotes."
  [db & [opts]]
  @(select-quotes db opts))

(defn by-id
  "Return a quote by id."
  [db id & [opts]]
  (first @(compose
           (select-quotes db opts)
           (where `(= :quotes.id ~id) :and))))

(defn by-symbol
  "Return a quote by symbol."
  [db symbol & [opts]]
  (first @(compose
           (select-quotes db opts)
           (where `(= (lower :quotes.symbol)
                      (lower ~symbol))
                  :and))))
