(ns antistock.sentiment
  (:import edu.stanford.nlp.pipeline.StanfordCoreNLP
           edu.stanford.nlp.ling.CoreAnnotations
           edu.stanford.nlp.sentiment.SentimentCoreAnnotations
           edu.stanford.nlp.ling.CoreAnnotations$SentencesAnnotation
           edu.stanford.nlp.sentiment.SentimentCoreAnnotations
           edu.stanford.nlp.neural.rnn.RNNCoreAnnotations)
  (:require [clojure.string :refer [blank? join]]))

(def sentences-annotation
  "The CoreMap key for getting the sentences contained by an annotation."
  edu.stanford.nlp.ling.CoreAnnotations$SentencesAnnotation)

(def sentiment-annotated-tree
  "A tree which contains the annotations used for the Sentiment task."
  edu.stanford.nlp.sentiment.SentimentCoreAnnotations$SentimentAnnotatedTree)

(defn human-class
  "Returns the human readable sentiment class."
  [class]
  (case class
    0 "very negative"
    1 "negative"
    2 "neutral"
    3 "positive"
    4 "very positive"
    "unknown"))

(defn pipeline
  "Make a Stanford Core NLP pipeline."
  [& {:keys [annotators]}]
  (let [annotators (or annotators [:tokenize :ssplit :parse :sentiment])]
    (StanfordCoreNLP.
     (doto (java.util.Properties.)
       (.setProperty "annotators" (join ", " (map name annotators)))))))

(defn sentences
  "Returns the sentences of the annotation."
  [annotation]
  (.get annotation sentences-annotation))

(defn tree
  "Returns the sentence tree of the annotation."
  [annotation]
  (.get annotation sentiment-annotated-tree))

(defn analyzer
  "Return an analyzer fn."
  [& opts]
  (let [pipeline (apply pipeline opts)]
    (fn [text]
      (if-not (blank? text)
        (if-let [annotation (.process pipeline text)]
          (for [sentence (sentences annotation)
                :let [tree (tree sentence)]]
            [(RNNCoreAnnotations/getPredictedClass tree)
             (str sentence)]))))))
