(ns antistock.worker.twitter
  (:require [antistock.db :as db]
            [antistock.db.test :refer [example-tweet]]
            [antistock.queue :as broker]
            [antistock.sentiment :as sentiment]
            [clj-http.client :as http]
            [clj-http.conn-mgr :as conn-mgr]
            [clojure.java.jdbc :as jdbc]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.exchange :as exchange]
            [langohr.queue :as queue]
            [no.en.core :refer [format-url parse-url]]
            [schema.core :as s])
  (:import [com.rabbitmq.client Channel]
           [sqlingvo.db Database]))

(def ^:dynamic *defaults*
  {:exchange "api"
   :prefetch 1
   :queue "worker.tweets"
   :routing-key "tweets.statuses.create"})

(s/defrecord Worker
    [exchange :- s/Str
     routing-key :- s/Str
     queue :- s/Str]
  {(s/optional-key :channel) Channel
   (s/optional-key :consumer-tag) s/Str
   s/Any s/Any})

(def Tweet
  {:id s/Int
   s/Any s/Any})

(def accepted-content-types
  #{"text/html" "text/plain"})

(defn accepted-content-type? [content-type]
  (some #(re-matches (re-pattern (str ".*" %1 ".*")) content-type)
        accepted-content-types))

(defn clean-url [url]
  (-> (parse-url url)
      (dissoc :query-params)
      (format-url)))

(defn tweet-urls [tweet]
  (-> tweet :entities :urls))

(defn analyze-sentiment [worker tweet]
  (->> (first (last (sort-by #(count (last %1))
                             ((:analyze worker) (:text tweet)))))
       (assoc tweet :sentiment)))

(defn fetch-url
  "Fetch the content of `url`."
  [worker url]
  (http/get
   (:url url)
   {:connection-manager (:connection-manager worker)
    :insecure? true
    :throw-exceptions false}))

(defn fetch-links [worker tweet]
  (update-in
   tweet [:entities :urls]
   #(doall (map (fn [url]
                  (merge url (fetch-url worker url)))
                %1))))

(defn save-link-mentions
  "Save the relationship between the `tweet` and the links mentioned."
  [db tweet]
  (update-in
   tweet [:entities :urls]
   #(doall (map (fn [response]
                  (let [content-type (str (get-in response
                                                  [:headers "content-type"]))
                        redirected-url (last (:trace-redirects response))]
                    (if (and (= 200 (:status response))
                             (accepted-content-type? content-type))
                      (let [link (->> {:url (clean-url redirected-url)
                                       :body (:body response)
                                       :content-type content-type}
                                      (db/save-link db))]
                        (db/save-links-tweet
                         db {:tweet-id (:id tweet)
                             :link-id (:id link)})
                        (merge response link))
                      response)))
                %1))))

(s/defn ^:always-validate save-hashtag-mentions
  "Save the relationship between the `tweet` and the hashtags mentioned."
  [db :- Database tweet :- Tweet]
  (update-in
   tweet [:entities :hashtags]
   #(doall (map (fn [hashtag]
                  (let [hashtag (db/save-twitter-hash-tag
                                 db {:name (:text hashtag)})]
                    (db/save-hash-tags-tweet
                     db {:tweet-id (:id tweet)
                         :hash-tag-id (:id hashtag)})
                    hashtag))
                %1))))

(s/defn ^:always-validate save-quote-mentions
  "Save the relationship between the `tweet` and the quotes mentioned."
  [db :- Database tweet :- Tweet]
  (assoc-in
   tweet [:entities :quotes]
   (doall (map (fn [quote]
                 (db/save-tweets-quote
                  db {:tweet-id (:id tweet)
                      :quote-id (:id quote)})
                 quote)
               (db/quotes-mentioned db (:text tweet))))))

(s/defn ^:always-validate save-user-mentions
  "Save the relationship between the `tweet` and the users mentioned."
  [db :- Database tweet :- Tweet]
  (update-in
   tweet [:entities :user-mentions]
   #(doall (map (fn [user]
                  (let [user (db/save-twitter-user db user)]
                    (db/save-tweets-user
                     db {:tweet-id (:id tweet)
                         :user-id (:id user)})
                    user))
                %1))))

(s/defn ^:always-validate save-tweet
  "Save the `tweet` to the database."
  [worker tweet]
  (let [db (:db worker)
        user-id (-> tweet :user :id)]
    (jdbc/with-db-transaction [db db]
      (db/save-twitter-user db (:user tweet))
      (db/save-twitter-tweet db (assoc tweet :user-id user-id))
      (->> (save-hashtag-mentions db tweet)
           (save-link-mentions db)
           (save-quote-mentions db)
           (save-user-mentions db)))))

(defn log-processed
  "Log the tweet as being processed."
  [tweet]
  (log/infof
   "[%s] %s (%s)" (str/upper-case (sentiment/human-class (:sentiment tweet)))
   (str/replace (str (:text tweet)) #"\r|\n" " ")
   (-> tweet :user :screen-name))
  tweet)

(s/defn ^:always-validate publish-tweet
  "Publish a tweet."
  [worker :- Worker tweet :- Tweet]
  (broker/publish-edn
   (:channel worker)
   (:exchange worker)
   (:routing-key worker)
   tweet))

(s/defn ^:always-validate process-tweet
  "Process the `tweet`."
  [worker :- Worker tweet :- Tweet]
  (->> (analyze-sentiment worker tweet)
       (fetch-links worker)
       (save-tweet worker)
       (log-processed)))

(s/defn ^:always-validate processor
  "Return the tweet processor."
  [worker :- Worker]
  (let [db (:db worker)]
    (fn [channel {:keys [delivery-tag]} tweet]
      (try (process-tweet worker tweet)
           (basic/ack channel delivery-tag)
           (catch Throwable e
             (log/error e "Can't process tweet.")
             (try (basic/reject channel delivery-tag)
                  (catch Throwable e
                    (log/error e "Can't acknowledge message."))))))))

(defn- connection-manager []
  (conn-mgr/make-reusable-conn-manager {:insecure? true}))

(s/defn ^:always-validate start-worker :- Worker
  "Start the Twitter worker."
  [{:keys [exchange prefetch routing-key queue] :as worker} :- Worker]
  (let [channel (channel/open (-> worker :broker :connection))
        _ (if prefetch (basic/qos channel prefetch))
        _ (queue/declare channel queue {:auto-delete false :durable true})
        worker (assoc worker :analyze (sentiment/analyzer))
        consumer-tag (broker/subscribe-edn channel queue (processor worker))]
    (exchange/fanout channel exchange {:durable true})
    (queue/bind channel queue exchange {:routing-key routing-key})
    (log/infof "Twitter worker started.")
    (assoc worker
           :channel channel
           :connection-manager (connection-manager)
           :consumer-tag consumer-tag)))

(s/defn ^:always-validate stop-worker :- Worker
  "Stop the Twitter worker."
  [worker :- Worker]
  (when-let [channel (:channel worker)]
    (when-let [consumer-tag (:consumer-tag worker)]
      (basic/cancel channel consumer-tag))
    (when-let [connection-manager (:connection-manager worker)]
      (conn-mgr/shutdown-manager connection-manager))
    (channel/close channel)
    (log/infof "Twitter worker stopped."))
  (dissoc worker :analyze :channel :connection-manager :consumer-tag))

(extend-protocol component/Lifecycle
  Worker
  (start [worker]
    (start-worker worker))
  (stop [worker]
    (stop-worker worker)))

(defn new-worker
  "Return a new Twitter worker."
  [config]
  (map->Worker (merge *defaults* config)))
