(ns antistock.worker.wikipedia.views
  (:require [antistock.db.languages :as languages]
            [antistock.db.quotes :as quotes]
            [antistock.db.wikipedia :as wikipedia]
            [antistock.json :as json]
            [antistock.queue :as broker]
            [clj-http.client :as http]
            [clj-time.coerce :refer [to-date to-date-time]]
            [clj-time.core :refer [month now year]]
            [clj-time.format :refer [formatters unparse]]
            [clojure.java.jdbc :as jdbc]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [langohr.basic :as basic]
            [langohr.channel :as channel]
            [langohr.exchange :as exchange]
            [langohr.queue :as queue]
            [schema.core :as s])
  (:import [com.rabbitmq.client Channel]
           [java.util Date]
           [sqlingvo.db Database]))

(def ^:dynamic *defaults*
  {:exchange "wikipedia"
   :prefetch 1
   :queue "worker.wikipedia.views"
   :routing-key "wikipedia.page.views"})

(s/defrecord Worker
    [exchange :- s/Str
     routing-key :- s/Str
     queue :- s/Str]
  {(s/optional-key :channel) Channel
   (s/optional-key :consumer-tag) s/Str
   s/Any s/Any})

(s/defn ^:always-validate page-view-url
  "Return the Wikipedia page view url for `title` and `time`."
  [language :- s/Str title :- s/Str time :- Date]
  (let [time (to-date-time time)]
    (format "http://stats.grok.se/json/%s/%04d%02d/%s"
            language (year time) (month time) title)))

(s/defn ^:always-validate parse-page-views :- {Date s/Num}
  "Parse the page views from `s` and return a map from date to the
  number of page views."
  [s :- s/Str]
  (reduce
   (fn [result [date views]]
     (assoc result (to-date (name date)) views))
   {} (:daily_views (json/read-string s))))

(s/defn ^:always-validate page-views :- {Date s/Num}
  "Return the page views for the Wikipedia page `title` in `language`
  at `time` and return a map from date to the number of page views."
  [language :- s/Str title :- s/Str time :- Date]
  (let [response (http/get (page-view-url language title time))]
    (parse-page-views (:body response))))

(s/defn ^:always-validate summary
  "Compute a summary map from `page-views`."
  [page-views :- {Date s/Num}]
  (let [dates (sort (keys page-views))]
    {:start (first dates)
     :end (last dates)
     :sum (apply + (vals page-views))}))

(defn- log-summary
  "Log the summary."
  [quote language {:keys [start end sum]}]
  (log/infof (str "Saved %d Wikipedia page views for %s and "
                  "language %s from %s until %s.")
             sum (:symbol quote) (:name language)
             (unparse (formatters :date) (to-date-time start))
             (unparse (formatters :date) (to-date-time end))))

(s/defn ^:always-validate save-page-views
  [db :- Database quote :- s/Any time :- Date]
  (when-let [page (first @(quotes/wikipedia-pages db quote))]
    (if (:title page)
      (reduce
       (fn [stats {:keys [iso-639-1] :as language}]
         (let [page-views (page-views iso-639-1 (:title page) time)]
           (jdbc/with-db-transaction [db db]
             (doseq [[date count] page-views]
               (wikipedia/save-wikipedia-page-view
                db {:count count
                    :date date
                    :language-id (:id language)
                    :page-id (:id page)})))
           (let [summary (summary page-views)]
             (log-summary quote language summary)
             (assoc stats (keyword iso-639-1) summary))))
       {} (languages/languages db))
      (log/warnf "Can't save page views for page %s: Page has no title."
                 (:id page)))))

(s/defn ^:always-validate handler
  "Return the Wikipedia page view handler."
  [worker :- Worker]
  (let [db (:db worker)]
    (fn [channel {:keys [delivery-tag]} {:keys [quote time] :as msg}]
      (try (save-page-views (:db worker) quote time)
           (basic/ack channel delivery-tag)
           (catch Throwable e
             (log/error e (str "Can't message." (pr-str msg)))
             (try (basic/reject channel delivery-tag)
                  (catch Throwable e
                    (log/error e "Can't acknowledge message."))))))))

(s/defn ^:always-validate start-worker :- Worker
  "Start the Wikipedia page view worker."
  [{:keys [exchange prefetch routing-key queue] :as worker} :- Worker]
  (let [channel (channel/open (-> worker :broker :connection))
        _ (if prefetch (basic/qos channel prefetch))
        _ (queue/declare channel queue {:auto-delete false :durable true})
        consumer-tag (broker/subscribe-edn channel queue (handler worker))]
    (exchange/fanout channel exchange {:durable true})
    (queue/bind channel queue exchange {:routing-key routing-key})
    (log/infof "Wikipedia page view worker started.")
    (assoc worker
           :channel channel
           :consumer-tag consumer-tag)))

(s/defn ^:always-validate stop-worker :- Worker
  "Stop the Wikipedia page view worker."
  [worker :- Worker]
  (when-let [channel (:channel worker)]
    (when-let [consumer-tag (:consumer-tag worker)]
      (basic/cancel channel consumer-tag))
    (channel/close channel)
    (log/infof "Wikipedia page view worker stopped."))
  (dissoc worker :channel :consumer-tag))

(extend-protocol component/Lifecycle
  Worker
  (start [worker]
    (start-worker worker))
  (stop [worker]
    (stop-worker worker)))

(defn new-worker
  "Return a new Wikipedia page view worker."
  [config]
  (map->Worker (merge *defaults* config)))
