(ns antistock.zooconf
  "Synchronize the Antistock configuration with ZooKeeper."
  (:gen-class)
  (:require [antistock.config.core :as config]
            [antistock.curator.core :as curator]
            [antistock.util :as util]
            [clojure.set :as set]
            [clojure.tools.logging :as log]
            [commandline.core :as cli]
            [environ.core :refer [env]]
            [inflections.core :refer [pluralize]]
            [no.en.core :refer [format-url parse-url]]
            [schema.core :as s]))

(def default-path
  "The default ZooKeeper path for the configuration."
  "/antistock/config")

(defn config
  "Return the curator config from `env` and `opts`"
  [env & [opts]]
  (curator/curator
   (or (some-> opts :zookeeper parse-url (set/rename-keys {:user :username}))
       (config/zookeeper env))))

(defn public-url
  "Return the public zookeeper url for `path`."
  [curator path]
  (-> (assoc curator :uri path :scheme :zookeeper)
      (format-url)))

(defn synchronize
  "Synchronize the `config` in Zookeeper using the `curator` framework."
  [env path opts]
  (curator/with-curator [curator (config env opts)]
    (log/infof "Synchronizing configuration with %s." (public-url curator path))
    (doseq [[key value] (config/config env)
            :let [path (str path "/" (name key))
                  bytes (curator/encode curator value)]]
      (when-not (curator/exists? curator path)
        (curator/create curator path value {:persistent? true}))
      (log/infof " * %s, %s." path (pluralize (count bytes) "byte")))
    (log/infof "Successfully synchronized configuration.")))

(defn clean
  "Remove the config from Zookeeper."
  [env path opts]
  (curator/with-curator [curator (config env opts)]
    (log/infof "Removing configuration from %s." (public-url curator path))
    (when (curator/exists? curator path)
      (curator/delete curator path))
    (log/infof "Successfully removed configuration.")))

(defn -main
  "Synchronize the Antistock configuration with Zookeeper."
  [& args]
  (util/log-banner)
  (log/info "            ⋆⋆⋆ ZOOKEEPER CONFIGURATION SYNC ⋆⋆⋆")
  (log/info "")
  (cli/with-commandline [[opts args] args]
    [[h help "Print this help."]
     [c clean "Clean the configuration in ZooKeeper."]
     [p path "The ZooKeeper synchronization path." :string "ZOOKEEPER-PATH"]
     [n no-schema "Disable schema validation for the configuration."]
     [Z zookeeper "The ZooKeeper URL." :string "ZOOKEEPER-URL"]]
    (if (:help opts)
      (cli/print-help "as configure")
      (let [path (or (:path opts) default-path)]
        (s/set-fn-validation! (or (not (:no-schema opts)) true))
        (if (:clean opts)
          (clean env path opts)
          (synchronize env path opts))))))
