(ns antistock.zooconf.watcher
  (:require [antistock.curator.core :as curator]
            [antistock.system :refer [with-component]]
            [com.stuartsierra.component :as component]
            [clojure.tools.logging :as log]
            [no.en.core :refer [parse-url]]
            [schema.core :as s])
  (:import antistock.curator.core.Curator
           [clojure.lang Atom IFn]
           [org.apache.curator.framework.recipes.cache
            NodeCache NodeCacheListener]))

(s/defrecord Watcher
    [constructor :- IFn
     curator :- Curator
     node-cache :- (s/maybe NodeCache)
     path :- s/Str
     component :- Atom]
  {s/Any s/Any})

(s/defn ^:private current-data :- s/Any
  "Return the current node cache data."
  [watcher :- Watcher]
  (some->> (some-> watcher :node-cache .getCurrentData .getData)
           (curator/decode (:curator watcher))))

(s/defn ^:private start-component
  "Start the component of `watcher`."
  [watcher :- Watcher]
  (some->> (current-data watcher)
           ((:constructor watcher))
           (component/start)
           (reset! (:component watcher)))
  watcher)

(s/defn ^:private stop-component
  "Stop the component of `watcher`."
  [watcher :- Watcher]
  (some-> watcher :component deref component/stop)
  (reset! (:component watcher) nil)
  watcher)

(s/defn ^:private start-watcher ; :- Watcher
  "Start the configuration watcher."
  [watcher :- Watcher]
  (if (:node-cache watcher)
    watcher
    (let [curator (component/start (:curator watcher))
          node-cache (NodeCache. (:connection curator) (:path watcher))
          watcher (assoc watcher :curator curator :node-cache node-cache)]
      (.addListener
       (.getListenable node-cache)
       (reify NodeCacheListener
         (nodeChanged [_]
           (when (current-data watcher)
             (log/infof "Configuration at %s changed, restarting ..."
                        (:path watcher))
             (-> (stop-component watcher)
                 (start-component))))))
      (.start node-cache true)
      (start-component watcher))))

(s/defn ^:private stop-watcher ; :- Watcher
  "Stop the configuration watcher."
  [watcher :- Watcher]
  (some-> watcher :node-cache .close)
  (some-> watcher :curator component/stop)
  (assoc (stop-component watcher) :node-cache nil))

(extend-type Watcher
  component/Lifecycle
  (start [watcher]
    (start-watcher watcher))
  (stop [watcher]
    (stop-watcher watcher)))

(s/defn watcher
  "Return a new ZooKeeper configuration watcher."
  [url constructor]
  (if-let [url (parse-url url)]
    (-> (map->Watcher
         {:component (atom nil)
          :constructor constructor
          :curator (curator/curator url)
          :path (:uri url)})
        (component/using [:curator]))
    (throw (ex-info "Can't parse ZooKeeper url." {:url url}))))

(defmacro with-watcher
  [[watcher-sym url constructor] & body]
  `(with-component [~watcher-sym (watcher ~url ~constructor)]
     ~@body))
