(ns arctype.client.router
  (:require
    [clojure.string :as string]
    [com.stuartsierra.component :as component]
    [reagent.core :as reagent]
    [route-parser]
    [schema.core :as S]
    [string-to-query-params]))

(def RouteState
  {:route [S/Str]
   :params (S/maybe {S/Any S/Any})})

(def RoutesConfig
  {;:no-history #{[S/Str]}
   ;:public #{[S/Str]}
   ;:public-default SingularRouteState
   :canonical {[S/Str] {:pattern S/Str
                        (S/optional-key :parser) S/Any}}})

(def Config
  {:routes RoutesConfig
   (S/optional-key :request) S/Any
   (S/optional-key :window) S/Any
   (S/optional-key :state-atom) S/Any}) ; atom to use for route state. May be passed in to manage the route state externally.

(defn- xform-matched-params
  [match]
  (->> (js->clj match)
       (map (fn [[k v]]
              [(keyword (string/replace k "_" "-")) v]))
       (into {})))

(defn- match-canonical
  "Returns [base-path params]"
  [canon pathname]
  (loop [try-routes canon]
    (if-let [[canonical-path {:keys [parser]}] (first try-routes)]
      (if-let [match (.match parser pathname)]
        {:route canonical-path 
         :params (xform-matched-params match)}
        (recur (rest try-routes)))
      nil)))

(defn- xform-make-params
  [clj-params]
  (->> clj-params
       (map (fn [[k v]]
              [(string/replace (name k) "-" "_") v]))
       (into {})
       (clj->js)))

(defn- make-canonical
  "Return canonical route path"
  [canon route params]
  (when-let [{:keys [parser]} (get canon route)]
    (.reverse parser (xform-make-params params))))

(defn parse-path
  [path-str]
  (remove empty? (string/split path-str #"/")))

(defn- translate-file-path
  [path]
  (loop [path path]
    (if-let [ent (first path)]
      (if (= "index.html" ent)
        (rest path)
        (recur (rest path)))
      path)))

(defn- parse-params
  [search]
  (if (some? search)
    (let [parsed (string-to-query-params (.substring search 1))]
      (js->clj parsed :keywordize-keys true))
    {}))

(defn- parse-request-location
  [{:keys [request routes]}]
  (let [location (aget request "_parsedUrl")
        pathname (aget location "pathname")
        canonical-match (match-canonical (:canonical routes) pathname)]
    (if (some? canonical-match)
      canonical-match
      {:route (vec (parse-path pathname))
       :params (parse-params (aget location "search"))})))

(defn- parse-window-location
  [{:keys [window routes]}]
  (let [location (.-location window)
        pathname (aget location "pathname")
        canonical-match (match-canonical (:canonical routes) pathname)]
    (if (some? canonical-match)
      canonical-match
      {:route (vec (parse-path pathname)) 
       :params (parse-params (aget location "search"))})))

(S/defn goto! :- RouteState
  [{:keys [state]} 
   route :- [S/Str] 
   params :- (S/maybe {S/Any S/Any})]
  (reset! state {:route route
                 :params params}))

(S/defn state :- RouteState
  [this]
  @(:state this))

(defrecord Router [config state]
  )

(defn- initial-state
  [{:keys [request window] :as config}]
  (let [state-atom (or (:state-atom config) (reagent/atom nil))
        initial-route (cond 
                        (some? request) (parse-request-location config)
                        (some? window) (parse-window-location config))]
    (swap! state-atom (fn [init-state]
                        (if (some? init-state)
                          init-state
                          initial-route)))
    state-atom))


(defn- routing
  [routes]
  (update routes :canonical
          (fn [canon]
            (->> canon
                 (map (fn [[k v]]
                        [k (assoc v :parser (route-parser. (:pattern v)))]))
                 (into {})))))

(S/defn create
  [config :- Config]
  (let [config (update config :routes routing)]
    (component/using 
      (map->Router 
        {:config config
         :state (initial-state config)})
      [])))
