(ns leiningen.csv-to-edn
  (:require [clojure.string :as str]
            [clojure.data.csv :as csv]
            [clojure.java.io :as io]))

(def dicts (atom nil))
(def edn-str "")

(defn deep-merge* [& maps]
  (let [f (fn [old new]
            (if (and (map? old) (map? new))
              (merge-with deep-merge* old new)
              new))]
    (if (every? map? maps)
      (apply merge-with f maps)
      (last maps))))

(defn deep-merge [& maps]
  (let [maps (filter identity maps)]
    (assert (every? map? maps))
    (apply merge-with deep-merge* maps)))

(defn read-csv [input-file]
  (with-open [in-file (io/reader input-file)]
    (doall
      (csv/read-csv in-file))))

(defn edn-to-str [e depth]
  (loop [ks (keys e)
         s ""]
    (if-let [k (first ks)]
      (let [tabs (apply str (repeat depth "\t"))
            k-str (str "\"" k "\"")
            v (get e k)]
        (if (map? v)
          (recur (rest ks) (str s tabs k-str " #js{\n" (edn-to-str v (inc depth)) tabs "}\n"))
          (recur (rest ks) (str s tabs k-str " \"" v "\"\n"))))
      s)))

(defn build-lang-files [dst-folder]
  (doseq [dict @dicts]
    (let [lang (key dict)]
      (with-open [out-file (io/writer (str dst-folder "/" lang ".cljs"))]
        (.write out-file ";This file is generated automatically. Please do not modify it!\n")
        (.write out-file (str "(ns audyx.app.languages." lang "\n"))
        (.write out-file "\t (:use-macros [purnam.core :only [obj]]))\n\n")
        (.write out-file "(def dict #js{\n")
        (.write out-file (edn-to-str (val dict) 1))
        (.write out-file "\n})")))))

(defn csv-to-edn
  [project & {:keys [src-file dst-folder keys-columns-count]
    :or {src-file "all.csv" dst-folder "./" keys-columns-count 1}}]
  (let [v (read-csv src-file)
        languages (filter #(not (empty? %)) (nth v 0))]
    (reset! dicts (zipmap languages (cycle [{}])))
    (loop [rows (rest v)
           langs (keys @dicts)
           i keys-columns-count]
      (cond
        (empty? rows) ()
        (empty? langs) (recur (rest rows) (keys @dicts) keys-columns-count)
        :else
          (let [row (first rows)
                word (first row)
                translation (nth row i)]
          (as->
            (str/split word #"\.") $
            (into [(first langs)] $)
            (assoc-in {} $ translation)
            (swap! dicts deep-merge $))
          (recur rows (rest langs) (inc i)))))
    (build-lang-files dst-folder)))
