(ns docker-connect.storage
  (:require [clojure.tools.logging :as log]
            [taoensso.faraday :as dyn]
            [environ.core :refer [env]]
            [docker-connect.util :refer :all])
  (:import com.amazonaws.auth.InstanceProfileCredentialsProvider))

(defonce storage-ctx (merge {}
                            (when (env :http-proxy-host)
                              {:proxy-host (env :http-proxy-host)
                               :proxy-port (read-string (env :http-proxy-port))})

                            (when (in-micros?)
                              {:provider (InstanceProfileCredentialsProvider.)
                               :endpoint (str "http://dynamodb." (env :dynamo-installed-table-region) ".amazonaws.com")})

                            (when (env :db-endpoint)
                              {:access-key ""
                               :secret-key ""
                               :endpoint (env :db-endpoint)})))

(defonce installed-table (or (env :dynamo-installed-table-name)
                             :installed))

(defonce tracking-id "meta:tracking")

(defn create-table []
  (dyn/create-table storage-ctx installed-table
                    [:id :s]
                    {:throughput {:read 10 :write 10}
                     :block? true}))

(defn check-metadata []
  (try
    (dyn/put-item storage-ctx installed-table
                  {:id tracking-id
                   :install-count 0
                   :uninstall-count 0}
                  {:expected {:id :not-exists}})
    (catch #=(dyn/ex :conditional-check-failed) e
           (log/info "Metadata already exists, skipping create"))))

(defn installed-count []
  (let [tracking (dyn/get-item storage-ctx installed-table {:id tracking-id})]
    (int (reduce - (keep tracking [:install-count :uninstall-count])))))

(defn- inc-installed []
  (dyn/update-item storage-ctx installed-table {:id tracking-id}
                   {:install-count [:add 1]}))

(defn- inc-uninstalled []
  (dyn/update-item storage-ctx installed-table {:id tracking-id}
                   {:uninstall-count [:add 1]}))

(defn init-storage []
  (log/info "Intitialising storage connection" (pps storage-ctx))
  (log/info "Installed table is" installed-table)

  (check-metadata))


;; FIXME: Ideally we should pull the necessary metadata out into the
;; top level of the record via a migration to simplify this. But this
;; isn't too onerous for now.
(defn app-ctx [client-key]
  (let [installed (dyn/get-item storage-ctx installed-table {:id client-key}
                                {:attrs [:context]})]
    (if (not (nil? installed))
      {:client-key client-key
       :key (get-in installed [:context "key"])
       :secret (get-in installed [:context "sharedSecret"])}
      nil)))

(defn shared-secret [client-key]
  ;; FIXME: Cache?
  (:shared-secret (dyn/get-item storage-ctx installed-table {:id client-key}
                                {:attrs [:shared-secret]})))

(defn save-install-context [context]
  (dyn/put-item storage-ctx installed-table
                {:id (context "clientKey")
                 :shared-secret (context "sharedSecret")
                 :install-username ((context "principal") "username")
                 ;; Dump of the context in-case we need it
                 :context (dyn/freeze context)})

  (inc-installed))

(defn delete-install-context [context]
  (dyn/delete-item storage-ctx installed-table
                   {:id (context "clientKey")})
  (inc-uninstalled))
