(ns clj-qobuz.playlist
  (:require [clj-qobuz.core :as core])
  (:refer-clojure :exclude [get]))

(defn get
  [playlist-id & [options]]
  (core/call-qobuz "playlist" "get" {:playlist_id playlist-id} options))

(defn user-playlists
  "Get playlists for user.
  `options` can contain `:query-params` with the following keys:
   * :username
   * :user_id
   * :order"
  [& [options]]
  (core/call-qobuz "playlist" "getUserPlaylists" {} options))

(defn create
  "Create playlist.
  `options` can contain `:query-params` with the following keys:
  * :description
  * :is_public (1 == public)
  * :is_collaborative (1 == collaborative)
  * :track_ids
  * :album_id (initial album to add to the playlist)."
  [name & [options]]
  (core/call-qobuz "playlist" "create" {:name name} options))

(defn delete-playlist
  "Delete a user playlist"
  [playlist-id & [options]]
  (core/call-qobuz "playlist" "delete" {:playlist_id playlist-id} options))

(defn update-playlist
  "Update a user playlist
  `options` can contain `:query-params` with the following keys:
  * :name
  * :description
  * :is_public (1 == public)
  * :is_collaborative (1 == collaborative)
  * :track_ids (tracks to replace the current tracks with)"
  [playlist-id & [options]]
  (core/call-qobuz "playlist" "update" {:playlist_id playlist-id} options))

(defn add-playlist-tracks
  "Add tracks to a user playlist"
  [playlist-id track-ids & [options]]
  (core/call-qobuz "playlist" "addTracks" {:playlist_id playlist-id
                                           :track_ids track-ids} options))

(defn delete-playlist-tracks
  "Remove tracks from a playlist - note that playlist_track_ids are not
  the track_ids, they are ids that come from the playlist items and
  are shown in the playlist_track_id field. The playlist_track_id will
  identify a specific instance of a track within a playlist (even when
  the same track_id is repeated)."
  [playlist-id playlist-track-ids & [options]]
  (core/call-qobuz "playlist" "deleteTracks" {:playlist_id playlist-id
                                              :playlist_track_ids playlist-track-ids} options))

;; See https://github.com/Qobuz/api-documentation/blob/master/endpoints/playlist/getFeatured.md
(defn featured-playlists
  "Get featured playlists.
  `listtype` must be one of `editor-picks` or `last-created`
  `options` can contain `:query-params` with the following keys:
  * :genre_ids list of genre ids"
  [listtype & [options]]
  (core/call-qobuz "playlist" "getFeatured" {:type listtype} options))
