(ns clj-qobuz.core
  (:require [cemerick.url :as url]
            [clj-http.client :as http]
            [clojure.string :as str])
  (:import java.security.MessageDigest))

(def ^:dynamic *endpoint* "https://www.qobuz.com/api.json/0.2")
(def ^:dynamic *app-id* "")
(def ^:dynamic *app-secret* "")

(defn signature-string
  [qobuz-object qobuz-method query-params timestamp app-secret]
  (let [sorted-params (->> query-params
                           (#(dissoc % :app_id :user_auth_token))
                           (sort-by first)
                           (map (fn [[k v]] (str (name k) v)))
                           str/join)]
    (str qobuz-object qobuz-method sorted-params timestamp app-secret)))

(defn hexify
  [bytes]
  (->> bytes (map (partial format "%02x")) str/join))

(defn md5
  [s]
  (let [d (MessageDigest/getInstance "MD5")]
    (.update d (.getBytes s))
    (hexify (.digest d))))

(defn signed-options
  [options]
  (let [{:keys [timestamp query-params qobuz-object qobuz-method app-id app-secret app-id token]
         :or   {timestamp  (quot (System/currentTimeMillis) 1000)
                app-id     *app-id*
                app-secret *app-secret*}} options

        query-params (if token (assoc query-params :user_auth_token token) query-params)
        sig-string   (signature-string qobuz-object qobuz-method query-params timestamp app-secret)
        sig          (md5 sig-string)]

    (-> options
        (assoc :query-params query-params)
        (assoc-in [:query-params :app_id] app-id)
        (assoc-in [:query-params :request_ts] timestamp)
        (assoc-in [:query-params :request_sig] sig))))

(defn qobuz-request
  [options]
  {:method       :get
   :url          (str (url/url (:endpoint options *endpoint*)
                               (:qobuz-object options)
                               (:qobuz-method options)))
   :query-params (:query-params options)
   :as           :json})

(defn call-qobuz
  ([options]
   (:body (http/request (qobuz-request (signed-options options)))))
  ([qobuz-object qobuz-method options]
   (call-qobuz (merge {:qobuz-object qobuz-object :qobuz-method qobuz-method} options)))
  ([qobuz-object qobuz-method query-params options]
   (call-qobuz qobuz-object qobuz-method (merge-with merge {:query-params query-params} options))))
