(ns bloomberg.data.index
  (:require
    [clj-time.core :as t]
    [bloomberg.request.reference :refer [bloomberg-get-reference-data]]
    [bloomberg.request.date :refer [str-2-date]]
    ))


(defn index-members- [list]
  (map #(str (:MemberTickerandExchangeCode %) " Equity") list))

(defn preprocess-index-response [response]
  (let [instrument (first response)
        members (index-members- (:INDX_MEMBERS instrument))
        members2 (index-members- (:INDX_MEMBERS2 instrument))
        members3 (index-members- (:INDX_MEMBERS3 instrument))]
    (-> instrument
        (clojure.set/rename-keys {:security :symbol
                                  :INDX_DES :description
                                  })
        (assoc :category "Index"
               :members (concat members members2 members3) )
        (dissoc :INDX_MEMBERS :INDX_MEMBERS2 :INDX_MEMBERS3)
        (update :INDEX_MEMBERSHIP_MAINT_DATE  str-2-date)
        (update :HISTORY_START_DT  str-2-date)
        )))


(defn get-index
  "gets index members"
  [bbsymbol]
  (-> {:symbol bbsymbol
       :field ["NAME"
               "HISTORY_START_DT"
               "MARKET_STATUS"; "ACTV"
               "SECURITY_TYP";  "Equity Index"

               "CUR_MKT_CAP";  1065669625000.0,
               "ID_BB_GLOBAL"
               "INDX_DES"
               "INDX_SOURCE"
               "INDX_UNITS"  ; quotation-type: [yield, discount rate, volume, number, quantity]
               "INDX_TYP"    ; 1: "Price Weighted"
                             ; 2: "Capitalization Weighted"
                             ; 3: "Equally Weighted"
                             ; 4: "Modified Capitalization Weighted"
                             ; 5: "Total Return"
                             ; 6: "Basket Weighted"
               "CRNCY";  "EUR"
               "INDEX_MEMBERSHIP_MAINT_DATE"    ; since when bb has memberships
               "INDX_FREQ"     ;Ranges from intraday, daily, weekly, monthly, quarterly or yearly.
               "INDX_HIST_START_DT_DAILY"
               "INDUSTRY_GROUP_INDEX"
               "MARKET_SECTOR" ; [1 - Commodity 2 - Equity 3 - Municipals 4 - Preferred 6 - Money Market 7 - Government 8 - Corporate 9 - Index 10 - Currency 11 - Mortgage]
               "INDX_DIVISOR"                            ;The Divisor is an arbitrary number chosen at the starting point of an index to
               ; fix the index starting value (example 1000.00). The divisor is then adjusted when capitalization
               ; amendments are made to the constituents of the index allowing the index value to remain comparable at all times.
               "COUNT_INDEX_MEMBERS"
               "INDX_MEMBERS"               ;  1-2500
               "INDX_MEMBERS2"                           ; 2500-5000
               "INDX_MEMBERS3"                           ; 5001-7500
               "INDX_MWEIGHT"                            ; Index Members Weights
               "INDX_GWEIGHT"]}
      (bloomberg-get-reference-data)
      (preprocess-index-response)))

(comment

  (get-index "DAX Index")
  (get-index "TRAN Index")

  ; parsing test
  (preprocess-index-response indx-response)

  ; INDEX MEMBERS
  (let [index-response (get-index-members "ATX Index") ; RIY(1000) RTY(2000)
         one (first index-response) ; old definition had a list.
         one index-response ]
    (println "Full Response: " index-response)
    (println "MEMBERS: ")
    (map println (:members one))
    ;(map #(println (str (:MemberTickerandExchangeCode %) " Equity")) (:members one))
   )

  )
