(ns bloomberg.fields
  (:require
    [clojure.string :as str]
    [clojure.java.io :as io]
    [clojure.java.shell :only [sh]]
    [cheshire.core :refer :all]                     ; JSON Encoding
    [clj-time.core :as t]
    [clj-time.format :as fmt]
    [mongo.db :refer [db]]
    [mongo.field]
    ))

; FIELDS

; 10|Real Time Quotes|0| |0007|Volume|VOLUME|1942|11|2
(defn process-field-info-line-
  "separates | delimited string columns into a map with fields as per definition"
  [fields line]
  (zipmap fields (str/split line #"\|")))

(defn load-bb-file [file-extension fields]
  (let [filename (str "resources/" "bbfields." file-extension)
        one (partial process-field-info-line- fields)]
    (if (.exists (io/as-file filename))
      (->> (slurp filename)
           (clojure.string/split-lines)
           (map one)))))

(defn import-bloomberg-fields
  "Imports Bloomberg TBL and EXT files into mongdb (so we have good field references)
   Obsolete Memonics are not imported, because they have the same field-id as the real memonics.
   Example:
   cat bbfields.tbl | grep 0D41
   152|Fundamentals - Cash Flow|0| |0D41|Inc(Dec) in ST Borrowings|CF_INCR_ST_BORROW|276|12|2
   999|Obsolete Mnemonic|0| |0D41|Inc(Dec) in ST Borrowings|EQY_CF027|0|12|2
  "
  []
  (let [tbl-all (set (load-bb-file "tbl" [:category-id :category-name :x :y :id :name :symbol :a :b :c]))
        tbl (remove #(= "999" (:category-id %)) tbl-all)
        xx (println "Saving TBL with " (count tbl) " rows (including obsolete: " (count tbl-all) ") ..")
        xx (doall (map #(mongo.field/save-field db %) tbl))
        ext (set (load-bb-file "ext" [:id :description]))
        xx (println "Saving EXT with " (count ext) " rows..")
        xx (doall (map #(mongo.field/save-field db %) ext))
        ]
    (println "import finished. Please look to mongdb FIELDS table")))

(comment

  ; FIELDS

  (str/split "asdf|444|asdf" #"\|")
  (zipmap [:symbol :b] ["asdf" 34])

  (println (first (load-bb-file "ext" [:id :description])))
  (println (first (load-bb-file "tbl" [:category-id :category-name :x :y :id :name :symbol :a :b :c])))

  (import-bloomberg-fields)

  )
