(ns bloomberg.request.reference
  (:require
    [clj-time.core :as t]
    [clojure.spec.alpha :as s]
    [clojure.test :refer :all]
    [bloomberg.request.request :refer [bloomberg-request-data]]
    [bloomberg.request.date :refer [str-2-date]]
    ))


(defn symbol-or-list-to-str
  "converts vector to ; separated string, or returns the string"
  [symbol]
    (if (string? symbol)
        symbol
        (clojure.string/join ";" symbol)))

(defn tap [v]
  (println v)
  v)

(defn append-override?-
  [request request-array]
  (let [override (:override request)]
    (if (nil? override)
        request-array
        (concat request-array (concat ["-o"] override))
    )))

;(append-override {:override ["YEAR=1" "MONTH=2"]} ["-s" "DAX Index" "-f" "NAME"])

(defn get-reference-data-raw
  "Makes bloomberg referebce request. Returns vector of maps"
  [request]
    (->> ["-s" (symbol-or-list-to-str (:symbol request))
          "-f" (symbol-or-list-to-str (:field request))]
         (append-override?- request)
         ;(tap)
         (bloomberg-request-data "Reference")
         (map #(clojure.set/rename-keys % {:security :symbol}))
         ))

;(s/def ::tuple (s/vec string? ))
;(s/conform ::tuple '[ "5" "1" "4" "5"])
;(s/valid? ::tuple '[ "5" "1" "4" "5"])

;[{category :- s/Str nil} {data :- s/Str nil}]

(s/def ::full-string (s/and string? #(> (count %) 0)))
(s/def ::symbol-or-list
   (s/or :single ::full-string
         :list vector?))

(s/def ::symbol ::symbol-or-list)
(s/def ::field ::symbol-or-list)
(s/def ::request-model (s/keys :req-un [::symbol ::field]))

(defn request? [request]
  (s/valid? ::request-model request))

(defn bloomberg-get-reference-data [request]
   (if (request? request)
      (get-reference-data-raw request)
      (throw (Throwable. (str "Wrong Bloomberg Request. Needed: {:symbol :field [:override]}"
                              (s/explain ::request-model request))))))

(def requests-bad [
  nil
  "DAX Index"
  {:s "DAX Index"}
  {:symbol "DAX Index"}
  ])

(def requests-ok [
  {:symbol "DAX Index" :field "PX_LAST"}
  {:symbol ["DAX Index" "ATX Index"] :field "PX_LAST"}
  {:symbol ["DAX Index" "ATX Index"] :field ["PX_LAST" "NAME"]}
  ])

(defn run-requests [requests expected-result]
  (map #(is (= expected-result (request? %) requests))))

(deftest test-request-bad
  (testing "BB ReferenceRequest - bad"
      (run-requests requests-bad false)))

(deftest test-request-ok
   (testing "BB ReferenceRequest - ok"
        (run-requests requests-ok true)))


(comment
  (run-tests 'bloomberg.request.reference)

  (bloomberg-get-reference-data {:symbol "DAX Index" :field ["PX_LAST" "NAME"]})

; Neuqueen 2025 bond."AN288649@BVAL Corp"
;
(bloomberg-get-reference-data {:symbol "AO0266368@BVAL Corp" :field [
   "PX_LAST"
   "CHG_NET_1M"
   "PX_CLOSE_DT"
   "COUPON"
   "MATURITY"
   "COLLAT_TYP"
   "ID_ISIN"
   ]})




(let [request {:symbol ["ATX Index" "DAX Index" "UQA AV Equity"] :field ["NAME" "PX_LAST" "HISTORY_START_DT"]}
      xx (println "Request Validity: " (request? request))
      response (get-reference-data-raw request)]
     (println response))

(comment [[
  {:security "ATX Index", :NAME "AUSTRIAN TRADED ATX INDX", :PX_LAST 2929.43}
{:security "DAX Index", :NAME "DAX INDEX", :PX_LAST 12167.56}]])





)
