(ns fundamental.metrics
  (:require
   [clojure.string :as str]
   [clj-time.core :as t]
   [taoensso.tufte :as tufte :refer (defnp p profiled profile)]

   [series.core :refer [load-series-params]]
   [series.splits :refer [load-split-data shares-outstanding shares-adjusted]]
   [fundamental.report :refer [report-end-date load-reports]]
   ))

;; add :report to timeseries

(defn skip-old-reports- [reports date]
  (drop-while #(t/after? (:releasedate %) date) reports))

(defn add-report- [reports-reverse series-element]
  (let [prior-reports (skip-old-reports- reports-reverse (:date series-element))]
    (assoc series-element :report (first prior-reports)
                          :report-prior (second prior-reports))))

(defn add-report-to-series
  "adds a :report field to series containing the most recent financial report"
  [series reports]
  (let [series-reverse (vec (reverse series))
        reports-reverse (vec (reverse reports))]
    (reverse (map (partial add-report- reports-reverse) series-reverse))))


; METRICS

(defn remove-nil-fields- [record]
  (apply merge (for [[k v] record :when (not (nil? v))] {k v})))

(defn sum-fields- [report fields]
  (let [fields-subset (select-keys report fields)
        fields-not-nil (remove-nil-fields- fields-subset)
        fields-vector (vals fields-not-nil)
        fields-sum (reduce + fields-vector)]
    fields-sum))

(defn divide [a b]
  (if (or (nil? a) (nil? b) (= 0 b) ) nil (/ a b)))


(defmacro swallow-exceptions [& body]
  `(try ~@body (catch Exception e#)))

(defn pos-or-nil [number]
  (if (> number 0) number nil))

(defn calc-metrics [splits price-field series-entry]
  (let [report (:report series-entry)
        report-fields (:fields report)
        price (price-field series-entry)
        turnover (divide (* (:volume series-entry) price) 1000000)
        empty-data  {:shares-adj nil
                     :market-cap nil

                     :sales nil
                     :price-sales nil
                     :sales-growth nil

                     :ebit nil
                     :price-ebit nil
                     :ebit-margin nil

                     :price-earnings nil
                     :dividend-yield nil
                     :price-book nil
                     :asset-yield nil

                     :ev nil
                     :ev-ebit nil

                     :debt-ebit nil
                     :turnover-mc nil
                     }
        report-prior (:report-prior series-entry)
        report-fields-prior (:fields report-prior)

        ]
    (if (not (or (nil? report) (nil? price)))
      (let [#_shares1 #_(:IS_SH_FOR_DILUTED_EPS  report-fields) ; more correct, but sometimes not available
            shares2 (:BS_SH_OUT report-fields)             ; fallback
            #_shares #_(if (nil? shares1) shares2 shares1 )
            shares shares2
            shares-out (if (nil? shares)
                           nil
                           (shares-outstanding shares (report-end-date report) splits (:date series-entry)))
            shares-adj (shares-adjusted shares-out splits (:date series-entry))
            marketcap (if (nil? shares-adj) nil (* price shares-adj))
            ]

        (if (not (nil? marketcap ))
          (assoc {}
            :shares-out shares-out
            :shares-adj shares-adj
            :market-cap marketcap

            :turnover (swallow-exceptions turnover)
            :turnover-mc (swallow-exceptions (divide turnover marketcap)  )

            :sales (swallow-exceptions (:SALES_REV_TURN report-fields))
            :price-sales (swallow-exceptions (divide marketcap (:SALES_REV_TURN report-fields)) )
            :sales-growth (swallow-exceptions (* 100 (- (divide (:SALES_REV_TURN report-fields) (:SALES_REV_TURN report-fields-prior)) 1.0))   )

            :ebit (swallow-exceptions (:IS_OPER_INC report-fields))
            :price-ebit (swallow-exceptions( pos-or-nil (divide marketcap (:IS_OPER_INC report-fields))))
            :ebit-margin (swallow-exceptions (* 100 (divide (:IS_OPER_INC report-fields) (:SALES_REV_TURN report-fields))))

            :price-earnings (swallow-exceptions (pos-or-nil (divide marketcap (:NET_INCOME report-fields))) )
            :dividend-yield (swallow-exceptions (pos-or-nil (* 100 (divide (:IS_TOT_CASH_COM_DVD report-fields)  marketcap)) ) )
            :price-book (swallow-exceptions (divide marketcap (sum-fields- report-fields [
                                                                :BS_RETAIN_EARN
                                                                :BS_SH_CAP_AND_APIC])))
            :asset-yield (swallow-exceptions (* 100 (divide (:IS_OPER_INC report-fields) (:BS_TOT_ASSET report-fields))))
            :ev (swallow-exceptions (+ marketcap (:BS_LT_BORROW report-fields)))
            :ev-ebit (swallow-exceptions  (pos-or-nil (divide (+ marketcap (:BS_LT_BORROW report-fields)) (:IS_OPER_INC report-fields))))
            :debt-ebit (swallow-exceptions  (pos-or-nil (divide (:BS_LT_BORROW report-fields) (:IS_OPER_INC report-fields))))
            )
           empty-data
          ))
      empty-data
      )))


(defn calc-metrics-safe [splits price-field series-entry]
   (try (calc-metrics splits price-field series-entry)
        (catch Exception e
          (println (str "caught exception: " (.getMessage e) price-field series-entry ) )
          nil)))


(defn add-metrics- [splits price-field series-entry ]
  (let [metrics (calc-metrics-safe splits price-field series-entry)]
    (assoc series-entry :report-metrics metrics)
    ))


(defn add-report-metrics-to-series
  "adds :report and :report-metrics field to series"
  [price-field series reports splits]
  (let [;_ (println "adding report to series..")
        series-with-report (add-report-to-series series reports)
        ;_ (println "adding metrics..")
        add-metrics-p (partial add-metrics- splits price-field)]
        (map add-metrics-p series-with-report)
        ))


;; LOAD FROM DB AND DO IT.

(defn calculate-metrics-series [symbol frequency]
  (let [reports-yearly (p :load-reports (doall (load-reports symbol)) )
        reports-sorted (sort-by :period reports-yearly)
        my-splits (load-split-data symbol)
        series (load-series-params {:frequency frequency
                                    :split-adjust true
                                    :data {:splits my-splits}
                                     } symbol)
        series-with-metrics (add-report-metrics-to-series :close series reports-sorted my-splits)]
    series-with-metrics))




(comment

   (swallow-exceptions (/ 1 0))

; Inject to series

  (def reports [
                {:releasedate (t/date-time 2007 03 01) :fields {:BS_SH_OUT 100 :SALES_REV_TURN 1000}}
                {:releasedate (t/date-time 2008 03 01) :fields {:BS_SH_OUT 100 :SALES_REV_TURN 1200}}
                {:releasedate (t/date-time 2009 03 01) :fields {:BS_SH_OUT 100 :SALES_REV_TURN 1300}}
                {:releasedate (t/date-time 2010 03 01) :fields { :SALES_REV_TURN 1400}} ; test if no BS_SH_OUT
                {:releasedate (t/date-time 2011 03 01) :fields {:BS_SH_OUT 90 :SALES_REV_TURN 1500}}
              ])  ; reports are orded by releasedate ascending

  (def series [ { :date (t/date-time 2000 12 30) :price 10.34} ; should link to report nil
               { :date (t/date-time 2009 12 30) :price 10.34}
               { :date (t/date-time 2010 01 30) :price 11.44}
               { :date (t/date-time 2010 02 28) :price 10.94}
               { :date (t/date-time 2010 03 30) :price 12.23}
               { :date (t/date-time 2010 04 30) :price 13.85}
               { :date (t/date-time 2015 04 30) :price 30.44} ; should link to last report from 2011
             ]) ; series is orded by releasedate ascending


  (skip-old-reports- (reverse reports) (t/date-time 2010 04 01))
  (add-report-to-series series reports)

  ( div-safe 3 nil)

; join (in sql terms vs. a report which I tend to think of as using aggregate
; operators like count or avg) in which case you should just clojure.core/for
; (for a cross product)

  (:output
   (reduce
    (fn [{:keys [report output] :as accum} thing]
      (if (contains? thing :releasedate)
        (assoc accum :report thing)
        (update-in accum [:output] conj (assoc thing :report report))))
    {:report nil  :output []}
    (sort-by ; merge sort if inputs are already sorted
     (fn [thing]
       (or (:releasedate thing)
           (:date thing)))
     (concat reports series))))


; METRICS

  (sum-fields- {:a 100 :b 10 :c 1} [:a :b])

  (def xx (add-report-to-series series reports))
  (def lr (last xx))
  (println lr)

  (calc-metrics :price lr)


  (->
   "BP/ LN Equity"
   ;"AAPL US Equity"
   ;"MSFT US Equity"
   (calculate-metrics-series :monthly)
   (last)
   ;(:open)
   ;(:report-metrics)
   ;(get-in [:report-metrics :price-sales])
   )





  )
