(ns fundamental.percentile
  (:require
    [clojure.tools.logging :as log]
    [com.stuartsierra.frequencies :as freq]
    [taoensso.tufte :as tufte :refer (defnp p profiled profile)]
    [mongo.stats]

    [fundamental.metrics :refer [calculate-metrics-series]]
    ))

(defn extract-single-metric-series [series metrics-name]
  (let [metrics-series (map #(get-in % [:report-metrics metrics-name]) series)
    ;_ (println metrics-name "has members: " (count metrics-series))
    ;_ (println metrics-series)
    metrics-series (remove nil? metrics-series)
    metrics-series (remove #(java.lang.Double/isNaN %) metrics-series)
    ;_ (println metrics-name "has members (without nil/NaN): " (count metrics-series))
    ]
    metrics-series))

(def percentile-definition
  [
   {:upper-bound 20 :valuation-state "very cheap"}
   {:upper-bound 40 :valuation-state "cheap"}
   {:upper-bound 60 :valuation-state "neutral"}
   {:upper-bound 80 :valuation-state "expensive"}
   {:upper-bound 100 :valuation-state "very expensive"}])

(defn valuation-state [stats current-value]
  (if (nil? current-value)
    nil
    (let [percentile-data (map #(assoc % :data (get-in stats [:percentiles (:upper-bound %)])) percentile-definition)
          highest-match (some #(if (<= current-value (:data %)) %) percentile-data)]
          ;(println "pd: " percentile-data)
          ;highest-match
          (if (nil? highest-match) nil (:valuation-state highest-match)))))



(defn metrics-stats [series metrics-name]
  (let [metrics-series (extract-single-metric-series  series metrics-name)
        freq-map (frequencies metrics-series)
        stats (freq/stats freq-map :percentiles (map :upper-bound percentile-definition)) ;[20 35 65 80]
        ]
    stats
    ))

(defn metrics-stats-all [series]
  { :price-sales (metrics-stats series :price-sales)
    :price-earnings (metrics-stats series :price-earnings)
    :price-ebit (metrics-stats series :price-ebit)
    :price-book (metrics-stats series :price-book)
    :dividend-yield (metrics-stats series :dividend-yield)
  })


(defn loosing-percentage [series]
    (let [                                                  ;xx (println series)
          ebit-series (map #(get-in % [:report-metrics :ebit]) series)
          ;xx (println ebit-series)
           all (remove nil? ebit-series)
           negative (filter neg? all)
          ;positive (filter pos? all)
           all-count (count all)
           neg-count (count negative)
          ;xx (println "neg: " neg-count " all: " all-count)
          ]
      (if (= 0 all-count)
        nil
        (* 100.0 (/ neg-count all-count))
        )
      ))



(defn history-ratio-sales [symbol]
  (let [series (calculate-metrics-series symbol :monthly)
        last-price-sales (get-in (last series) [:report-metrics :price-sales])
        stats (metrics-stats series :price-sales)
        stats-sales-growth (metrics-stats series :sales-growth)
        ]
    (log/info symbol " P/S: " last-price-sales "stats: " stats)
    {:symbol            symbol
     :current           last-price-sales
     :current-valuation (valuation-state stats last-price-sales)
     :stats             stats
     :loosing-year-prct (loosing-percentage series)
     ; :stats-sales-growth stats-sales-growth
     :median-sales-growth (:median stats-sales-growth)
     }
    ))

(defn calc-history-and-save [symbol]
  (let [stats (history-ratio-sales symbol)
        x (println "stats are: " stats)
        x (mongo.stats/save-stats stats)
        x (println "saving done..")]
    {:calc-stats 1}
    ))

(comment

  (into (vector) {:a 1 :c 3 :b 2})

  ; calculate cheap/very-cheap/neutral/expensive/very-expensive
  (valuation-state {:percentiles {20  1.5
                                  35  3.0
                                  65  6.0
                                  80  12.0
                                  100 34.4}} 1.3)

  ; Calculation

  (history-ratio-sales "BP/ LN Equity")
  (run fundamental.percentile/calc-history-and-save "OK List" index)

  ; Performance Tracking
  (tufte/add-basic-println-handler! {})
  (profile {} (p :history (history-ratio-sales "BP/ LN Equity")))
  (profile {} (p :history (calc-history-and-save  "BP/ LN Equity")))



  )
