(ns job.run
  (:require
    [clojure.tools.logging :refer [info]]
    [clj-time.core :as t]
    [clojure.data :as cljdata]
    [taoensso.tufte :as tufte :refer (defnp p profiled profile)]
    [db.config :refer [config]]
    [mongo.db :refer [db]]
    [mongo.instrument :refer :all]
    [mongo.universe :refer [browse-equities]]
    [mongo.series]
    [mongo.cost]
  ))


; PIPE Functions:
; - Pipe Data from Bloomberg into Database
; - First Parameter: db
; - Second Parameter: symbol

; SYMBOL FUNCTIONS

(defn index [symbol]
  (->> symbol
       (mongo.instrument/load-symbol)
       (:members)))

(defn equities [symbol]
  (->> (browse-equities {})
       (map #(:symbol %))))

(defn symbols [symbol]
   symbol)


(defn timeseries [_]
   (->> (mongo.series/symbols-with-timeseries
            (-> 11 t/years t/ago)
            (-> 1 t/years t/ago)) ;(t/now)   
        (map :symbol)
     ))



(defn all-equity-symbols-from-db [symbol]                ; symbol is irrelevant
  (let [equities (mongo.instrument/load-equities)
        equities-no-name (filter #(nil? (:NAME %)) equities)
        symbols (map :symbol equities-no-name)]
    symbols))

; RUNNER


; {:start #object[org.joda.time.DateTime 0x49aa995d "2019-01-09T16:59:09.555Z"],
; :end   #object[org.joda.time.DateTime 0x10ae124 "2019-01-09T18:04:35.797Z"],
; :work  {:report 2103, :cost 8412}}

(def bloomberg-cost-table
  {:index-desc     1
   :equity-desc    2                                        ;fields: 15
   :splits         1
   :report-summary 1
   :report         4                                        ; fields: 60
   :series         0.005                                    ; 1 year = 200 bars = 1 cost
   })

(defn get-bloomberg-cost
  "returns cost per bloomberg job
   or 0 if not found"
  [bloomberg-job-type]
  (let [cost (bloomberg-job-type bloomberg-cost-table)]
    (if (nil? cost) 0 cost)))


(defn calculate-cost-for-job [result]
  (let [cost (map #(* (get % 1) (get-bloomberg-cost (get % 0))) result)
        ;xx (println cost)
        ]
    (reduce + cost)
    ))


(defn run-function [func symbol]
  (let [result (func symbol)
        cost (calculate-cost-for-job result)
        stats (conj result {:cost cost})
        xx (mongo.cost/save-cost cost)
        ]
    stats
    ))


(defn run-function? [func symbol]
  (let [current-cost (mongo.cost/todays-cost)
        cost-limit (get-in config [:bloomberg :daily-cost-limit])
        ;xx (println "current cost: " current-cost)
        ;xx (println "cost limit: " cost-limit)
        ]
    (if (< current-cost cost-limit)
      (run-function func symbol)
      {:cost-limited 1}
      )
    ))


(defn aggregate-results [results]
  (let [ xx (println "aggregate-results " results)
  ]
    ;(apply merge-with + results))
    (reduce (partial merge-with +) {} results)
    ))



(defn run-jobs [f symbols]
  (let [start (t/now)
        results (doall (map f symbols))
        end (t/now)
        result (aggregate-results results)
        ]
    {:start start
     :end   end
     :work  result}
    ))

(defn run [function-data symbol & [function-symbols]]
  (let [ f (partial run-function? function-data)
        #_xx #_(println "function-symbols: " function-symbols)
        symbols (if (nil? function-symbols)
                  [symbol]
                  (function-symbols symbol))
        ;xx (println "symbols are: " symbols)
        x (println "Running on " (count symbols) " symbols..")
        ]
    (run-jobs f symbols)
    ))


(comment

  (def demo-results [ {:bond 1 :series 3} {:equity 10 :series 20}])


  (merge-with + {:bond 1 :series 3} {:equity 10 :series 20})
  (reduce (partial merge-with +) {} demo-results)
  (merge-with + demo-results)
;(apply merge-with + results)

  (aggregate-results [{:bond 0, :cost 0} {:bond 0, :cost 0} {:bond 0, :cost 0}
                      {:bond 0, :cost 0} {:bond 0, :cost 0} {:bond 0, :cost 0}
                      {:cost 0} {:bond 0, :cost 0} {:bond 0, :cost 0}
                      {:bond 0, :cost 0} {:bond 0, :cost 0} {:bond 0, :cost 0}
                      {:bond 0, :cost 0}])


  ; Job Cost
  (get-bloomberg-cost :unknown-job)
  (get-bloomberg-cost :report)
  (calculate-cost-for-job {:report 100 :b 10 :series 2000})

  ; symbol generation functions
  (index "RTY Index")

  ; Dummy JOB
  (defn bongo [symbol]
    (println "BONGO CALCULATION - " symbol)
    {:report 1})
  (run-function bongo "XX")
  (run-function? bongo "XX")

  ; Job Runner
  (run bongo "LHA GY Equity")
  (run bongo "DAX Index" index)




  )                                                         ; ****************************************************
