(ns mongo.db
  (:require
   [clojure.tools.logging :refer [info]]
   [mount.core :as mount]
   [clj-ssh.cli :as sshcli]
   [clj-ssh.ssh :as ssh]
   [environ.core :refer [env]]
   [clj-time.core :as t]
   [monger.core :as mg]
   [monger.collection :as mc]
   [monger.joda-time])
  (:import [com.mongodb MongoOptions ServerAddress]))


; MOUNT SERVICE DEFINITION

(defn make-tunnel [remote-ip remote-port local-port private-key username]
  (info "creating ssh tunnel to" remote-ip ":" remote-port)
  (let [agent (ssh/ssh-agent {})]
    (ssh/add-identity agent {:private-key private-key})
    (let [session (ssh/session agent remote-ip {:username username :strict-host-key-checking :no})]
      (ssh/connect session
        (ssh/forward-local-port session local-port remote-port)))))

(def config (atom {:mongo-ip "127.0.0.1"
                   :mongo-port 27017
                   :mongo-db "seasonality"}))

(reset! config {:mongo-ip "127.0.0.1"
                          :mongo-port 29017
                          :mongo-db "seasonality"
                          :tunnel {:ip "aws.hoertlehner.com"
                                   :remote-port 27017
                                   :private-key (slurp "/home/andreas/Documents/myLinux/ssh/flo5")
                                   :username "admin" }})

(defn connect []
  (let [;trading-env (env :trading-env)
        ;port (if (nil? trading-env) 27017 29017) ; 27017 default port if no trading-env defined. 29017 (ssh tunnel) if it is defined
        tunnel (:tunnel @config)
        _ (when tunnel (make-tunnel (:ip tunnel) (:remote-port tunnel) (:mongo-port @config) (:private-key tunnel) (:username tunnel)))
        conn (mg/connect {:host (:mongo-ip @config) :port (:mongo-port @config)}) ; 29017 on client via tunnel
        _ (info "connecting database port " (:mongo-port @config))
        db (mg/get-db conn (:mongo-db @config))]
    {:db db :conn conn}))


(defn disconnect [conn-db]
  (info "disconnecting database..")
  (mg/disconnect (:conn conn-db)))

(mount/defstate db-conn
  :start (connect)
  :stop  (disconnect db-conn))

(mount/defstate db
  :start (:db db-conn))

; ACTIONS

(defn in?
  "true if coll contains elm"
  [coll elm]
  (some #(= elm %) coll))

(defn capitalize-category [data]
  (update data :category clojure.string/capitalize))

(defn separate-category [instrument]
  (->> instrument
       (:symbol)
       (re-matches #"(.*)\s(\w+)\s*")
       (drop 1)
       (zipmap [:symbol :category])
       (capitalize-category)
       (merge instrument)
       (#(assoc % :symbol (str (:symbol %) " " (:category %))))))

; (defn load-symbol [db symbol]
;  (mc/find-one-as-map  db "instruments" { :symbol symbol }))

(comment ; *************************************************************************

; test db connection
  (do
    (def test- (connect))
    (print test-)
    (println (mc/find-one-as-map (:db test-) "instruments" {:symbol "DAX Index"}))
    (disconnect test-))

; environment variable is important because it effects the port of mongodb
  (let [penv #(println %1 "=" (env %1))]
    (do
      (penv :shell) ; /bin/bash
      (penv :home) ; /home/Florian
      (penv :user) ; florian, admin
      (penv :dummy-dummy)
      (penv :trading-env))) ; dev on dev machine via starttunnel.sh used by mount

; use db services (requires started mount services)
  (mount/start)
  (println (mc/find-one-as-map db "instruments" {:symbol "DAX Index"}))

  (separate-category {:symbol "MO equity " :name "jghf "})
  (in? ["Equity" "Index" "Curncy"] (:category {:category "Index" :bonho 88}))) ; ********************************************************************************
