(ns mongo.report
  (:require
    [clojure.tools.logging :refer [info]]
    [clj-time.core :as t]
    [clojure.string :as str]
    [monger.core :as mg]
    [monger.collection :as mc]
    [monger.joda-time]
    [mongo.db :refer [db]]
    [mongo.setting :refer [load-setting]]
;    [data.report :refer [extract-year-type-number]]
    ))


(defn save-report
  "Saves one Financial Report to db. Old Report for same symbol/period will be replaced"
  [report]
  ;( let [sanitized-report (merge report (extract-year-type-number report))]
  ;  (mc/update db "reports" {:symbol (:symbol report) :period (:period report)} sanitized-report {:upsert true})))
  (mc/update db "reports" {:symbol (:symbol report) :period (:period report)} report {:upsert true}))

(defn modify-report
  "Modifies one Financial Report to db. Old Report for same symbol/period will be replaced"
  [report]
  (println "Modifying Report" "Symbol:" (:symbol report) "Period:" (:period report))
  (mc/update db "reports" {:symbol (:symbol report) :period (:period report)} {:$set report} {:upsert true}))


(defn load-reports-for-symbol
  "load reports (all / only of type) for a symbol"
  ([symbol]
    (mc/find-maps db "reports" {:symbol symbol}))
  ([symbol type]
    (mc/find-maps db "reports" {:symbol symbol :type type})))

(defn load-reports-available-for-symbol [symbol]
  "array of report-period (available for symbol)"
  (map :period
       (mc/find-maps db "reports" {:symbol symbol} {:period 1})))

(defn load-annual-reports-available-for-symbol [symbol]
  "array of report-period (available for symbol)"
  (map :period
       (mc/find-maps db "reports" {:symbol symbol :type "A"} {:period 1})))

; FIELDS [Get fields that we are interested in]

(defn get-report-fields-full
  "load report-fields from settings (including unwanted fields)"
  []
  (let [setting (load-setting "reportfields")
        all-fields (:data setting)
        fields-with-symbol (remove #(nil? (:symbol %)) all-fields )
        fields-sorted (sort-by :position fields-with-symbol)
        fields-with-symbol (map #(assoc {} (keyword (:symbol %)) %) fields-sorted )
        ]
    ;(into (sorted-map) fields-with-symbol)
    fields-sorted
    ))

(defn get-report-fields
  "load report-fields from db (filtered only wanted fields)"
  []
  (let [setting (load-setting "reportfields")
        all-fields (:data setting)
        fields-with-symbol (remove #(nil? (:symbol %)) all-fields )
        report-symbols (map :symbol fields-with-symbol) ]
    (map keyword report-symbols)
    ))

(defn available-report-overview 
  [& [symbol]]
  (let [q (if (nil? symbol)
            {}
            (if (vector? symbol)
              {:symbol {:$in symbol}}
              {:symbol symbol} ))] 
    
    (->> (mc/aggregate db "reports"
                       [{:$match q}
                        {:$group {:_id  { :symbol "$symbol"} ; :type "$type"}
                                  :annual {:$sum {:$cond {:if {:$eq ["$type","A"] }  :then 1 :else 0}} }
                                  :quarterly {:$sum {:$cond {:if {:$eq ["$type","Q"] }  :then 1 :else 0}} }
                                  :semia {:$sum {:$cond {:if {:$eq ["$type","S"] }  :then 1 :else 0}} }
                                  :y2009 {:$sum {:$cond {:if {:$eq ["$year",2009] }  :then 1 :else 0}} }
                                  :total {:$sum 1}}}
                        {:$sort {:_id 1}}
             ;{:$project {:symbol "$_id.symbol"} }
                        ])
         (map #(assoc % :symbol (get-in % [:_id :symbol])))
         (map #(dissoc % :_id) )              
         )))

(defn change-symbol [symbol-old symbol-new]
  (mc/update db "reports"
             {:symbol symbol-old}
             {:$set {:symbol symbol-new}}
             {:upsert false :multi true}))
    




(comment

  ; FIELDS - all and of interested [together with display settings]
  (get-report-fields-full)
  (load-setting "reportfields")
  (def fields- (get-report-fields ))
  (println fields-)

  ; INDIVIDUAL FINANCIAL REPORTS
  (count (load-reports-for-symbol "A2A IM Equity"))
  (count (load-reports-for-symbol  "A2A IM Equity" "A"))
  (load-reports-available-for-symbol  "MCD US Equity")

  (save-report {:symbol "MO XXX" :period "bongo" :xxx "hzzz"})

  (available-report-overview)

  (available-report-overview "CAI AV Equity")
  (available-report-overview ["CAI AV Equity" "BP/ LN Equity"])
  
  )
