(ns mongo.universe
  (:require
    [clojure.tools.logging :refer [info]]
    [mount.core :refer [defstate]]
    [environ.core :refer [env]]
    [clj-time.core :as t]
    [monger.core :as mg]
    [monger.collection :as mc]
    [monger.joda-time]
    [mongo.db :refer [db]]
    [clojure.pprint :as pp]
    ))


(defn all-instruments []
  (mc/find-maps db "instruments" {}))



; INDEX **************************************************

(defn browse-indices [category]
  (mc/find-maps db "instruments" {:category category} {:symbol 1 :NAME 1 :COUNT_INDEX_MEMBERS 1}))

(defn browse-indices-no-members []
  (mc/find-maps db "instruments"
                {:category "Index"
                 :members {:$size 0}}
                {:symbol 1 :NAME 1 :COUNT_INDEX_MEMBERS 1}))




(defn index-member-summary [symbol]
  (mc/aggregate
    db
    "instruments"
    [{:$match {:symbol symbol}}
     {:$unwind "$members"}
     {:$lookup {:from         "instruments",
                :localField   "members",
                :foreignField "symbol",
                :as           "i"}}
     {:$group {:_id                 '$_id'
               :description         {:$first "$description"}
               :INDX_SOURCE         {:$first "$INDX_SOURCE"}
               :NAME                {:$first "$NAME"}
               :COUNT_INDEX_MEMBERS {:$first "$COUNT_INDEX_MEMBERS"}
               :i                   {:$push "$i"}}}
     {:$project {
                 "_id"                 0
                 "symbol"              1
                 "NAME"                1
                 "description"         1
                 "INDX_SOURCE"         1
                 "COUNT_INDEX_MEMBERS" 1
                 "i.symbol"            1
                 "i.name"              1}}
     ])
  )








; EQUITY **************************************************************

(defn browse-equities
  ([search]
   (->> (mc/find-maps db "instruments"
                 (merge {:category "Equity"} search)
                 {:symbol      1
                  :exchange    1
                  :name        1
                  :sector      1
                  :industry    1
                  :subindustry 1})
       (map #(dissoc % :_id))
       ;(sort-by (juxt :sector :industry :subindustry :name)
       ))
  ([]
   (browse-equities {})))


(defn equity-groups [field]
  (mc/aggregate
    db
    "instruments"
    [{:$match {:category "Equity"}}
     {:$group {:_id   field
               :count {:$sum 1}}}
     {:$sort {:_id 1}}
     ;{:$project {:_id 1 } }
     ])
  )

(defn equity-sector []
  (equity-groups "$sector"))

(defn equity-industry []
  (equity-groups "$industry"))

(defn equity-sub-industry []
  (equity-groups "$subindustry"))



(defn sector-info []
  (->> (mc/aggregate
        db
        "instruments"
        [{:$match {:category "Equity"}}
         {:$group 
          {:_id {:sector "$sector"
                 :industry "$industry"
                 :subindustry "$subindustry"}
           :count {:$sum 1} }}
         {:$sort {:_id 1}}
       ;{:$project {:_id 1 } }
         ])
       (map #(assoc (:_id %) :count (:count %)) )

       
       (sort-by :subindustry)
       (sort-by :industry)
       (sort-by :sector)
       
       ))






(defn equity-exchange []
  (equity-groups "$exchange"))

(comment

  (count (browse-indices "List"))
  (count (browse-indices "Index"))
  (count (browse-indices-no-members ))

  (count (browse-equities ))
  (count (browse-equities  {:sector "Utilities"}))

  (pp/print-table (browse-equities  {:sector "Utilities"}))
  (pp/print-table (browse-equities {
                                    :name {:$ne nil}
                                    :sector {:$ne nil}
                                    :industry {:$ne nil}
                                    :subindustry {:$ne nil}
                                    }))

  (browse-equity-groups {:name {:$ne nil}
                         :sector {:$ne nil}
                         :industry {:$ne nil}
                         :subindustry {:$ne nil}})

  (pp/print-table [:sector :industry :subindustry :count] (sector-info))
  
  (take 5 (browse-equities ))

  (equity-sector )
  (equity-industry )
  (equity-sub-industry )
  (equity-exchange )

  (index-member-summary  "DAX Index")

  
  )
