(ns seasonal.stats
  (:require
    [clojure.string :as str]
    [clj-time.core :as t]
    [series.core :refer [load-series]]
    [seasonal.math :as math]
  ))

; ** MONTH NAME **

(defn getMonth [x] (t/month (x :date)))
(defn getYear [x] (t/year (x :date)))

(defn add-month
  "add month field to the timeseries"
  [timeseries]
  (map #(assoc %
          :month (getMonth %)
          :year (getYear %)
          :year-month (+ (* (getYear %) 100) (getMonth %))
          ) timeseries))


; ** PERCENTAGE CHANGE **

(defn prct-change
  "calculates percentage change of prior/current"
  [prior current]
  (do
    ;(println "current" current "prior" prior)
    (* (/ (- current prior) prior) 100.0)))                    ; absolute change


; X Month change calculation

(defn add-change-month
  "adds field close price change field to timeseries"
  [changeName prior current]
  (assoc current changeName (prct-change (:close prior) (:close current)))  ; :close-adj
  )

(defn timeseries-add-change-x
  [months-ago timeseries]
  "calculates changes in closing price (relative to x months ago)"
  (map #(add-change-month (keyword (str "chg-" months-ago)) %1 %2)
       timeseries                                           ; prior
       (drop months-ago timeseries)))                       ; current


(def timeseries-add-change-1-m
  (partial timeseries-add-change-x 1))



(defn add-change-month-prior
  "adds field close price change field to timeseries (based on prior month ending)"
  [changeName current prior current-minus-one]
  (assoc current changeName (prct-change (:close prior) (:close current-minus-one)))) ;:close-adj


(defn shift-series
  "shift timeseries by x elements"
  [field timeseries]
  ;  (map field timeseries )
  (map #(assoc %2 field (field %1)) timeseries (drop 1 timeseries))
  )




(defn timeseries-add-change-prior-x
  [months timeseries]
  (map #(clojure.set/rename-keys %1 {(keyword (str "chg-" months)) (keyword (str "chg-p-" months))})
       (shift-series (keyword (str "chg-" months)) (timeseries-add-change-x months timeseries))))

(def timeseries-add-change-p-1
  (partial timeseries-add-change-prior-x 1))

(def timeseries-add-change-p-3
  (partial timeseries-add-change-prior-x 3))

(def timeseries-add-change-p-6
  (partial timeseries-add-change-prior-x 6))

(def timeseries-add-change-p-12
  (partial timeseries-add-change-prior-x 12))




(defn get-transformed-series
  "load series then add month and price-change"
  ([symbol]
     (get-transformed-series symbol (load-series symbol :monthly)) )
  ([symbol series]
     (->> series
        (add-month)
        (timeseries-add-change-1-m)
        (timeseries-add-change-p-3)
        (timeseries-add-change-p-6)
        (timeseries-add-change-p-12)
        ; (partial timeseries-add-change-x 5 :chg-5)
        )))

(defn clone-inc-month [last-month]
  (if (< (:month last-month) 12)
    (update last-month :month inc)
    (assoc last-month :year (inc (:year last-month)) :month 1)))

(defn add-future-month
  [series]
  (if (or (nil? series) (= (count series) 0))
    []
    (let [last-month (last series)
          future-month (clone-inc-month last-month)]
      (conj series future-month))))



(defn get-transformed-series-future
  [symbol]
  "load series then add month and price-change"
  (->> (load-series symbol :monthly)
       (add-month)
       (vec)
       (add-future-month)
       (timeseries-add-change-1-m)
       (timeseries-add-change-p-3)
       (timeseries-add-change-p-6)
       (timeseries-add-change-p-12)
       ;  ( partial timeseries-add-change-x 5 :chg-5)
       ))


(defn uptrend?
  [field element]
  (> (field element) 0))

(def uptrend3? (partial uptrend? :chg-p-3))
(def uptrend6? (partial uptrend? :chg-p-6))
(def uptrend12? (partial uptrend? :chg-p-12))

(defn group-summary [months]
  (let [monthly-returns (map :chg-1 months)
        nr (count months)]
    (if (> nr 1)
      {:count  (count months)
       :mean   (math/mean monthly-returns)
       :stddev (math/standarddev monthly-returns)
       }

      {:count  0
       :mean   0.0
       :stddev 0.0}
      )))


(defn month-stats
  [tuple]
  (zipmap [:month :all :up-12 :down-12 :up-6 :down-6 :up-3 :down-3]
          [(get tuple 0)
           (group-summary (tuple 1))
           (group-summary (filter uptrend12? (tuple 1)))
           (group-summary (remove uptrend12? (tuple 1)))
           (group-summary (filter uptrend6? (tuple 1)))
           (group-summary (remove uptrend6? (tuple 1)))
           (group-summary (filter uptrend3? (tuple 1)))
           (group-summary (remove uptrend3? (tuple 1)))
       ]))

(defn year-inside [options series-entry]
  (let [year (:year series-entry)]
    (and (>= year (:year-start options))
          (or (= (:year-end options) 0)
              (<= year (:year-end options))))))

(defn year-filter [options series]
  (if (nil? options)
  series (filter (partial year-inside options) series)))


(defn get-series-stats
  [symbol & [options]]
  (->> (get-transformed-series symbol)
       (year-filter options)
       (group-by :month)
       (map month-stats)
       ))

; optimal series stats

(defn total-stddev [month-stat]
  (reduce + (map :stddev (vec (vals (select-keys month-stat [:up-3 :down-3 :up-6 :down-6 :up-12 :down-12]))))))

(defn total-stddev-stats [month-stats]
  (reduce + (map total-stddev month-stats)))


(defn year-remove [year series]
  (remove #(= (:year %) year) series))

(defn calc-only-stats [series]
  (->> series
       (group-by :month)
       (map month-stats)))

(defn get-series-stats-optimal [symbol & [options]]
  (let [series (year-filter options (get-transformed-series symbol))
        years (map :year series)]
    (if (= (count years) 0)
        []
        (let [; if timeseries is empty, don't waist time in optimization
              ;xxx (println symbol years)
              year-start (apply min years)
              year-end (apply max years)
              ;xxx (println year-start "-" year-end " count: " (count series))
              year-series (map #(assoc {} :year % :series (year-remove % series)) (range year-start (+ 1 year-end)))
              year-stats (map #(assoc {} :year (:year %) :stats (calc-only-stats (:series %))) year-series)
              year-stats-with-total (map #(assoc % :total-stddev (total-stddev-stats (:stats %))) year-stats)
              ;zzz (println (map :total-stddev year-stats-with-total))
              year-stats-sorted (sort-by :total-stddev year-stats-with-total)
              best-year-stat (first year-stats-sorted)]
          best-year-stat
              ;(map #(total-stddev (:stats %)) year-stats)
         ))))


(defn stats-1-month [stats month]
   (first (filter #(= (:month %) month) stats)))

(comment

  (get-transformed-series "MSFT US Equity")

  (def ts (get-transformed-series "DAX Index"))

  (def ts (get-transformed-series-future "DAX Index"))
  (def ts (get-transformed-series-future "MUV2 GY Equity"))
  (println ts)

  (map #(println (:year %) (:month %)) (get-transformed-series-future  "DAX Index"))
  (map #(println (:year %) (:month %)) (get-transformed-series "DAX Index"))
  (map :month (get-transformed-series "DAX Index"))
  (year-remove 2018 (get-transformed-series "DAX Index"))

  (stats-1-month 11 (get-series-stats "DAX Index"))
  (get-series-stats-optimal "DAX Index")
  (get-series-stats-optimal "MAC US Equity")





  (group-summary (timeseries-add-change-1-m dummy-data))

  (timeseries-add-change-p-1 dummy-data)
  (timeseries-add-change-p-3 dummy-data)
  (timeseries-add-change-p-12 dummy-data)


  (uptrend? :chg-p-12 {:chg-p-12 0.5})
  (uptrend? :chg-p-12 {:chg-p-12 -0.5})
  (uptrend? :chg-p-12 {:chg-p-12 0.0})

  (def test [{:a 1 :d "jan"} {:a 2 :d "feb"} {:a 3 :d "march"} {:a 4 :d "april"} {:a 5 :d "may"}])
  (shift-series :a test)

   ; test percentage change
  (prct-change 200 240)

  ; adds change data to one month
  (add-change-month :chg-demo {:close 3} {:close 5})



  (take 5 (get-series-stats "SX3P Index" {:year-start 2000 :year-end 2010}))
  (get-series-stats "VNA GY Equity" {:year-start 2000 :year-end 2010})
  (get-series-stats-optimal "VNA GY Equity" {:year-start 2000 :year-end 2010})


(get-series-stats-optimal "DAX Index" {:year-start 0, :year-end 0, :max-positions 1, :min-risk-reward 30, :is-future false})
(load-series "DAX Index" :monthly)


  )
