(ns seasonal.trading
  (:require
    [clojure.pprint :as pp]
    [mongo.instrument :refer [load-symbol]]
    [seasonal.stats :as stats]
    [seasonal.print :as print]))

(defn instrument-calc
    "calculate all data for one symbol"
    [symbol & [options]]
    (let [is-future? (if (or (nil? options) (nil? (:is-future options))) false (:is-future options)) ]
     { :symbol symbol
       :series (doall (stats/get-transformed-series-future symbol)) ; future added here
       ;:stats (stats/get-series-stats symbol options)
       :stats (:stats (stats/get-series-stats-optimal symbol options))
     }))


; TREND RULES
; trend :up :down
; trade-type TREND/CYCLE
; trade-dir  :long :short :cash
; pl-expected

(defn trend-up-rule
   "trade-decision for uptrend, based on uptrend-average"
   [group]
   (if (nil? group) nil
      (if (> (:mean group) 0) { :trend :up    :trade-type :trend   :trade-dir :long    :pl-expected (:mean group) :stddev (:stddev group) :count (:count group)  }
                 { :trend :up    :trade-type :cycle   :trade-dir :short   :pl-expected (- 0 (:mean group)) :stddev (:stddev group) :count (:count group) }
        )))

(defn trend-down-rule
  "trade-decision for downtrend, based on downtrend-average"
   [group]
   (if (nil? group) nil
   (if (> (:mean group) 0) { :trend :down   :trade-type :cycle  :trade-dir :long    :pl-expected (:mean group) :stddev (:stddev group) :count (:count group) }
                 { :trend :down   :trade-type :trend  :trade-dir :short   :pl-expected (- 0 (:mean group)) :stddev (:stddev group) :count (:count group) }
  )))



(defn stats-for-month
  "find statistics for a given month"
   [stats month]
   (first (filter #( = (:month %) month)  stats)))


(defn wrap-symbol-data
   [symbol trade-decision]
   ( assoc trade-decision :symbol symbol ))




(defn realized-pl
   "pl of trade long/short"
   [timeseries-entry direction-stats]
     (if (= (:trade-dir direction-stats) :long) (:chg-1 timeseries-entry) (- 0 (:chg-1 timeseries-entry)) )
  )


(defn trade-for-lookback [month-stats timeseries-entry symbol rule-number up-field down-field trend-rule?]
   (let  [direction-stats ( if (trend-rule? timeseries-entry)
                               (trend-up-rule (up-field month-stats))
                               (trend-down-rule (down-field month-stats)))
          trade-suggestion (assoc direction-stats
                                :symbol symbol
                                :pl-realized (realized-pl timeseries-entry direction-stats)
                                :year (:year timeseries-entry)
                                :month (:month timeseries-entry)
                                :rule-number rule-number)]
          trade-suggestion))


(defn trade-signal-timeseries-entry
     "Converts TimeseriesEntry to trading signal"
     [ stats symbol timeseries-entry]
     ( let [ month-stats (stats-for-month stats (:month timeseries-entry))
             trade-3 (trade-for-lookback month-stats timeseries-entry symbol 3 :up-3 :down-3 stats/uptrend3?)
             trade-6 (trade-for-lookback month-stats timeseries-entry symbol 6 :up-6 :down-6 stats/uptrend6?)
             trade-12 (trade-for-lookback month-stats timeseries-entry symbol 12 :up-12 :down-12 stats/uptrend12?)
       ]
       [ trade-3 trade-6 trade-12]))


(defn trade-signal-timeseries
    [instrument]
    (map #(trade-signal-timeseries-entry (:stats instrument) (:symbol instrument) % )  (:series instrument))
  )


(defn trade-symbol-multiple
   [symbol & [options]]
   ( let [ instrument (doall (instrument-calc symbol options))
            trades (trade-signal-timeseries instrument) ]
        (flatten (doall trades))
    ))


(def year-month-vector
  (for [year (range 1993 2019)
        month (range 1 13)]
    [year month]))

(defn trades-by-risk-reward
   [trades]
   (let  [trades-not-nil (remove nil? trades)
          trades-ok (filter #(contains? % :count) trades-not-nil)
          trades-count-min (filter #( >= (:count %) 5) trades-ok)
          trades-risk-reward-min (filter #( >= (:pl-expected %)  (* 0.5 (:stddev %))) trades-count-min)
          trades-risk-reward (map #(assoc % :risk-reward (- (:pl-expected %) (* 0.5 (:stddev %)))
                                            :risk-reward-ratio (* (/ (:pl-expected %) (:stddev %)) 100.0)
                                    ) trades-risk-reward-min)
          trades-sorted ( reverse (sort-by :risk-reward trades-risk-reward )) ]
          trades-sorted
     ))


(defn best-trade-per-symbol-only [trades]
     (->> trades
       (map-indexed #(assoc %2 :index %1))
       (group-by (juxt :symbol :trade-dir))
       (mapv (fn [[[symbol trade-dir] trials]]
               (assoc (apply max-key :risk-reward trials)
                 :symbol symbol
                 :trade-dir trade-dir)))
       (sort-by :index)
       (mapv #(dissoc % :index))))


(defn trade-management-year-month
   [year-month trades options]
   (let [min-risk-reward (if (or (nil? options) (nil? (:min-risk-reward options))) 30 (:min-risk-reward options) )
         trades-sorted ( ->> trades
                             (remove nil?)
                             (filter #( and (= (:year %) (get year-month 0)) (= (:month %) (get year-month 1) )) )
                             (trades-by-risk-reward)
                             (filter #(>= (:risk-reward-ratio %) min-risk-reward) )
                             )
         ;x (println "trade management " year-month " min-risk-reward: " (count trades-sorted) )
         trades-symbol-trade-once (best-trade-per-symbol-only trades-sorted)
         ;x (println "trade management once per symbol: " (count trades-symbol-trade-once) )
         trades-sorted-long  (filter #(= (:trade-dir %) :long) trades-symbol-trade-once)
         trades-sorted-short (filter #(= (:trade-dir %) :short) trades-symbol-trade-once)
         max-positions (if (or (nil? options) (nil? (:max-positions options))) 1 (:max-positions options) )
         ;xxx (println "all: "(count trades-sorted))
         ;xxx (println trades-sorted)
         ;xxx (println year-month "all: " (count trades-sorted) " long: " (count trades-sorted-long) " short: " (count trades-sorted-short))
        ]
     (vec (concat (take max-positions trades-sorted-long) (take max-positions trades-sorted-short)))
     ))

(defn trade-management [trades options]
  (as-> trades y
        (do
          (println "trade management running on " (count y) " trades..") y)
          (pmap #(trade-management-year-month % y options) year-month-vector) ; pmap as symbol selection takes time
          (flatten y)
          (remove nil? y)))


(defn trade-symbol [symbol & [options]]
    (trade-management (trade-symbol-multiple symbol options) options))


(defn trade-symbol-multiple-and-log [symbol & [options]]
  ( do (println "calculating " symbol)
       (trade-symbol-multiple symbol options)))

(defn trade-symbols
  [symbols & [options]]
  (let [trade-symbols (pmap #(trade-symbol-multiple-and-log % options) symbols) ; pmap
        trades-all (apply concat trade-symbols) ]
    (trade-management trades-all options)
  ))

(defn add-cum-pl [trades]
  (let [trades-not-empty (remove nil? trades)
        pl (map :pl-realized trades-not-empty)
        cum-pl (reductions + pl)
        zipped (map vector trades cum-pl) ]
       (map #( assoc (get % 0) :cum-pl (get % 1))  zipped)
    ))


; VIEWs (was prior in web/views)

(defn trade-symbol-cum-pl [name & [options]]
  [name]
  (let [ trades (if (nil? name) [] (trade-symbol name options))
         ; xx (println trades)
         trades-cum-pl (add-cum-pl trades)]
    trades-cum-pl))

(defn trade-list-cum-pl [name & [options]]
  [name]
  (let [members (if (nil? name) [] (:members (load-symbol name) )  )
        trades (if (nil? name) [] (trade-symbols members options))
        trades-cum-pl (add-cum-pl trades)]
  trades-cum-pl))


; REPL TESTS
(comment ; *******************************************************************************

(vec (concat (take 2 [2 2 2]) (take 2 [3 3 3]) (take 2 [7 ]) ))

(take 1 [ 1 2 ])
(first [ 1 2])
(take 10 [])

(first [{:risk-reward-ratio 0.9823501780759155, :trend :down, :trade-dir :short, :pl-expected 3.7503966607284727,
          :stddev 3.817779794242216, :symbol "SXDP Index", :risk-reward 1.8415067636073648, :month 2, :trade-type :trend,
          :rule-number 3, :year 1993, :pl-realized 3.5450523882951335, :count 9}])

(trend-up-rule -0.4)
(trend-down-rule 0.4)

(trade-symbol "DAX Index")
(trade-symbol "AAPL Equity")
(trade-symbol "DAX Index" {:year-start 2000 :year-end 2010 :min-risk-reward 70})

(trade-symbols ["DAX Index" "SX86P Index" "AEX Index"])
(trade-symbols ["DAX Index" "SX86P Index"] {:year-start 2000 :year-end 2010 :is-future false})

(println x)
(add-cum-pl x)

(wrap-symbol-data "DAX" {:a "asdf"})

(doall year-month-vector)

(def mytrades[
  {:pl-expected 20 :year 2015 :month 3 :stddev 5 :count 5}
  {:pl-expected 50 :year 2015 :month 3 :stddev 5 :count 5}
  {:pl-expected 200 :year 2015 :month 4 :stddev 5 :count 5}])

(trades-by-risk-reward mytrades)

(>= 0.5 0.2)

(get-trade-for-year-month [2015 3] mytrades )

(print/print-trades(trade-symbols ["DAX Index" "SX86P Index"]))

) ; *******************************************************************************


(comment ; TEST ONE TRADE PER SYMBOL

(def v [ { :symbol "a" :rule-number "lqs1" :algo-params { :x 34 :y 34} :risk-reward 5 }
  { :symbol "a" :rule-number "nn1"  :algo-params  { :x 34 :y 34} :risk-reward 3 }
  { :symbol "a" :rule-number "lqs1" :algo-params { :x 4 :y 20} :risk-reward 2  :comment "remove this"}
  { :symbol "a" :rule-number "nn1" :algo-params { :x 34 :y 34} :risk-reward 1 :comment "remove this"}
  { :symbol "b" :rule-number "lqs1" :algo-params { :x 34 :y 34} :risk-reward 5 }
  { :symbol "b" :rule-number "nn1"  :algo-params  { :x 34 :y 34} :risk-reward 3 }
  { :symbol "b" :rule-number "lqs1" :algo-params { :x 4 :y 20} :risk-reward 2 :comment "remove this"}
  { :symbol "b" :rule-number "nn1" :algo-params { :x 34 :y 34} :risk-reward 1 :comment "remove this"}
  { :symbol "c" :rule-number "lqs1" :algo-params { :x 34 :y 34} :risk-reward 5 }
  { :symbol "c" :rule-number "nn1"  :algo-params  { :x 34 :y 34} :risk-reward 3 }
  { :symbol "c" :rule-number "lqs1" :algo-params { :x 4 :y 20} :risk-reward 2 :comment "remove this"}
  { :symbol "c" :rule-number "nn1" :algo-params { :x 34 :y 34} :risk-reward 1 :comment "remove this"}
])


(->> v
  (map-indexed #(assoc %2 :index %1))
  (group-by (juxt :symbol :rule-number))
  (mapv (fn [[[symbol rule-number] trials]]
          (assoc (apply max-key :risk-reward trials)
            :symbol symbol
            :rule-number rule-number)))
  (sort-by :index)
  (mapv #(dissoc % :index)))

) ; ***********************************************
