(ns series.splits
  (:require
    [clj-time.core :as t]
    [clojure.tools.logging :refer [info]]
    [mongo.instrument :refer [load-symbol]]
    ))

; STRING/DATE <=> INT-YEAR-MONTH conversions

(defn sdate-to-year-month
  "converts string YYYY-MM-DD to integer YYYYMMDD"
  [sdate]
  (let [syear (subs sdate 0 4)
        smonth (subs sdate 5 7)
        sday (subs sdate 8 10)
        year (. Integer parseInt syear)
        month (. Integer parseInt smonth)
        day (. Integer parseInt sday)
        ]
    (+ (* year 10000) (* month 100) day)))

(defn date-to-year-month
  "date => integer YYYYMMDD"
  [date]
  (let [year (t/year date)
        month (t/month date)
        day (t/day date)
        ]
    (+ (* year 10000) (* month 100) day)))


; LOAD FROM DB

(defn load-split-data
  "Loads Split data from db for a symbol and normalizes it,
   so it can be used in calculations"
  [symbol]
  (let [splits-raw (:EQY_DVD_HIST_SPLITS (do (load-symbol symbol)))]
   (if (nil? splits-raw)
       []
       (->> splits-raw
            (filter #(= (:DividendType %) "Stock Split"))
            (map #(assoc {} :year-month (sdate-to-year-month (:Ex-Date %)) :factor (:DividendAmount %)))
            (sort-by :year-month)))))

; TIMESERIES PRICE ADJUSTMENT FACTORS

(defn split-factor
  "Returns the split factor for a data (represented as int year-month)
   given the (normalized) split array
  "
  [splits year-month]
  (let [factor (->> splits
                    (filter #(> (:year-month %) year-month))
                    (map :factor)
                    (reduce *))]
    (do ;(println "adjusting " year-month " by:" factor )
      factor)))

(defn convert-field [bar field func factor]
  (let [val (field bar)]
    (if (nil? val)
        bar
        (assoc bar field (func val factor)) )))

(defn adjust-bar- [splits bar]
  (let [year-month-day (:year-month bar)
        year-month-day (if (nil? year-month-day)
                           (date-to-year-month (:date bar))
                            year-month-day)
        factor (split-factor splits year-month-day)]
    (-> bar
        (assoc :adj-fact factor)
        (convert-field :open / factor)
        (convert-field :high / factor)
        (convert-field :low / factor)
        (convert-field :close / factor)
        (convert-field :volume * factor))))

(defn timeseries-split-adjust-via-splits
   "split-adjusts timeseries with provided splits"
   [timeseries splits]
     (map #(adjust-bar- splits %) timeseries))

(defn timeseries-split-adjust
  "split-adjusts a timeseries"
  [timeseries symbol]
  (let [splits (load-split-data symbol)]
    (timeseries-split-adjust-via-splits timeseries splits)))


; SHARES ADJUSTMENT FACTORS

(defn report-correction-factor
  "The shares of the last report are what matters.
   However, if splits happen, then the time between split and the next report
   "
  [report-date-ym splits date-ym]
  (->> splits
       (filter #(> (:year-month %) report-date-ym))
       (filter #(<= (:year-month %) date-ym))
       (map :factor)
       (reduce *)))


(defn shares-outstanding
  "actual number of shares outstanding on a given date."
  [report-shares report-date splits date]
  (let [report-date-ym (date-to-year-month report-date)
          date-ym (date-to-year-month date)]
       #_(println "share-qty-split factor for" date-ym " is: " factor)
       (* report-shares (report-correction-factor report-date-ym splits date-ym))))

(defn shares-adjusted
  "share quantity adjusted for splits (so that shares can be visualized)"
  [shares-outstanding splits date]
    (* shares-outstanding (split-factor splits (date-to-year-month date))))


(comment

  ; Date conversions
  (sdate-to-year-month "1997-08-18T00:00:00")
  (some #(if (= % 1) %) '(3 4 1))

  ;  split test data

  (let [mysplits [{:year-month 19900501 :factor 3.0}
                     {:year-month 20021201 :factor 10.0}]
        print-factor #(println "factor in" %1 ":" (split-factor mysplits %1))]
       (share-qty-split-adjusted 2.0 19950101 mysplits 20010101)
       (print-factor 19700101)
       (print-factor 19800101)
       (print-factor 19850101)
       (print-factor 19980101)
       (print-factor 20100101)
       (print-factor 20200101)
       (println mysplits))

(timeseries-split-adjust [{:year-month 19900101 :close 5.5}
                          {:year-month 19970801 :close 7.5}
                          {:year-month 19980501 :close 7.5}
                          {:year-month 20100501 :close 7.5}]
      "EMSN SE Equity")

(macroexpand '(as->  5 x
                     8 y
                     (* x 3)
                     (/  100 x)
                     (* 1000 x y)
                        ))

  (convert-field {:open 100 :x 10} :open * 3)
  (convert-field {:open 100 :x 10} :close * 3)
  (convert-field {:open 100 :volume 3 :x 10} :volume / 3)


(-> 14 t/days t/ago)

; load split conversion factors
(->> ;"IIA AV Equity"
     "BP/ LN Equity"
     (load-split-data)
     (println))

 (timeseries-split-adjust [
   {:date (t/date-time 1998 1 2) :close 808.0, :high 813.0, :low 805.0, :open 805.5, :volume 2541941.0}
   {:date (t/date-time 1990 5 31) :close 330.0, :high 330.0, :low 323.0, :open 326.0}
   ]
   "BP/ LN Equity")


; load timeseries, and split adjust it
(future (let [symbol "BP/ LN Equity"
             ;"IIA AV Equity"
              summary #(info "processed " (count %) " items.")
     ]
     (-> symbol
         (mongo.series/load-series)
         ;(mongo.series/load-series-partial (-> 14 t/days t/ago) (t/now))
         ;(mongo.series/load-series-partial (t/date-time 1989 1 1) (t/date-time 1990 1 3))
         (timeseries-split-adjust symbol)
         (summary))))



  )
