(ns web.routes.tradingview
  (:require
    [clojure.tools.logging :refer [info]]
    [clj-time.core :as t]
    [clj-time.coerce :as c]
    [clojure.string :as str]

    [schema.core :as s]
    [ring.util.response :as response]
    [ring.util.http-response :refer [ok]]
    [compojure.core :refer :all] ; [compojure.core :refer [defroutes routes]]
    [compojure.route :as route]
    [compojure.api.sweet :as sweet]
    [cheshire.core :refer :all]
    [cemerick.url :refer (url url-encode map->query)]

    [import.series :refer [daily-series-latest]]

    [tradingview.config :refer [config ]]
    [tradingview.search :refer [tradingview-search-request]]
    [tradingview.symbol :refer [tradingview-symbol-info-request doublepoint-to-space]]
    [tradingview.series :refer [server-time tradingview-history]]
    [tradingview.storage :refer [load-chart save-chart modify-chart delete-chart]]
    [tradingview.template :refer [load-template save-template delete-template]]


    ;[tradingview.marks]
    ;[tradingview.quotes]
    ;[tradingview.group]
    ))

(s/defschema Chart
     {:symbol String
      :resolution String
      :name  String
      :content String
      ;:id     Long
     ;:price Double
     ;;:hot   Boolean
    ;;(s/optional-key :description) String
     })

(s/defschema Template
    {:name String
     :content String})

(defn save-chart-wrapped [client user name content symbol resolution]
  (let [data {:name name :content content :symbol symbol :resolution resolution}]
     {:status "ok" :id (save-chart client user data)}))

(defn modify-chart-wrapped [client user chart name content symbol resolution]
   (let [data {:name name :content content :symbol symbol :resolution resolution :chart_id chart}]
     (modify-chart client user chart data)
     {:status "ok"}))

(defn save-or-modify-chart [client user chart name content symbol resolution]
  (if (= chart 0)
       (save-chart-wrapped   client user       name content symbol resolution)
       (modify-chart-wrapped client user chart name content symbol resolution)))


(defn handler-rt-redirect [forward-to {params :params uri :uri body :body}]
  (let [sub-route (:* params)
        params-clean (dissoc params :*)
        querystring (map->query params-clean)
        destination (str/join [forward-to sub-route])
        destination (if (nil? querystring) destination (str/join [destination "?" querystring]))
  ]
     (println "rt redirect uri:" uri " params: " params " sub-route" sub-route)
     (println "redirecting to: " destination)
     (response/redirect destination)
  ))



(def routes-redirect-history
   ; Certain Routes go to the realtime version
   (GET "/api/tradingviewrt/history" [] (partial handler-rt-redirect "/api/tradingview/rt/history")))

(def routes-redirect-other
   ; All Other Routes go to the static version
   (GET "/api/tradingviewrt/*" [] (partial handler-rt-redirect "/api/tradingview/"))
)



(def add-routes
  (sweet/context "/api" [] :tags ["tradingview"]

    (sweet/context "/tradingview/rt" [] :tags ["tradingview"]
        (sweet/GET "/history" []
          :query-params [symbol :- String resolution from :- Long to :- Long]
          (do
            (let [symbol-clean (doublepoint-to-space symbol)]
              (daily-series-latest symbol-clean)
              (ok
                 (tradingview-history symbol-clean resolution from to))
            )
          ))
    )


    (sweet/context "/tradingview" [] :tags ["tradingview"]
        ;  (GET "/timeseries" [symbol] (generate-string (map #(dissoc % :date) (stats/get-transformed-series mongo/db  symbol))))
        ; https://demo_feed.tradingview.com/history?symbol=AAPL&resolution=D&from=1567457308&to=1568321308

        (sweet/GET "/config" []
            :query-params []
            (ok config))

        (sweet/GET "/time" []
            :query-params []
            :return Long
            (ok (server-time)))

        (sweet/GET "/search" []
            :query-params [query :- String
                           type :- String
                           exchange :- String
                           limit :- Long]
            (ok (tradingview-search-request query type exchange limit)))

        (sweet/GET "/symbols" []
            :query-params [symbol :- String]
            (ok (tradingview-symbol-info-request symbol)))

        (sweet/GET "/history" []
           :query-params [symbol :- String resolution from :- Long to :- Long]
           (ok (tradingview-history symbol resolution from to))))


        (sweet/context "/tradingviewstorage" [] :tags ["tradingview"]

        ; storage
        (sweet/GET "/1.1/charts" []
           :query-params [client :- s/Int user :- s/Int {chart :- s/Int 0}]
           (ok (if (= chart 0)
                   {:status "ok" :data (load-chart client user)}
                   {:status "ok" :data (load-chart client user chart)})))

        (sweet/POST "/1.1/charts" []
           :query-params [client :- s/Int user :- s/Int {chart :- s/Int 0}  ]
           ;:return Pizza
           :consumes ["application/x-www-form-urlencoded"]
           ;:body [chart-data Chart]
           ;:body-params  [chart-data s/Any]
           :form-params [name content symbol resolution]
           ;(ok {:status "ok" :id (save-chart client user chart-data)} ))
           (ok (save-or-modify-chart client user chart name content symbol resolution)))

        (sweet/PUT "/1.1/charts" []
           :query-params [client :- s/Int user :- s/Int {chart s/Int} ]
           ;:return Pizza
           :body [chart-data Chart]
           (ok (do (modify-chart client user chart chart-data)
                   {:status "ok"})))

        (sweet/DELETE "/1.1/charts" []
              :query-params [client :- s/Int user :- s/Int {chart :- s/Int 0} ]
              ;:return Pizza
              ;:body [data]
              (ok (do (delete-chart client user chart) {:status "ok"})))


        ; storage
        (sweet/GET "/1.1/study_templates" []
            :query-params [client :- s/Int user :- s/Int {chart :- s/Int 0}]
            (ok (if (= chart 0)
                 {:status "ok" :data (load-template client user)}
                 {:status "ok" :data (load-template client user client)})))

        (sweet/POST "/1.1/study_templates" []
            :query-params [client :- s/Int user :- s/Int  ]
              ;:return Pizza
            ;:body [template-data Template]
            :form-params [name content]
               (ok (do (save-template client user {:name name :content content})
                       {:status "ok"} )))
              ;(ok {:status "ok" :id (save-template client user template-data)} ))

        (sweet/DELETE "/1.1/study_templates" []
            :query-params [client :- s/Int user :- s/Int {template :- s/Str ""} ]
              (ok (do (delete-template client user template) {:status "ok"})))




  )))
